package io.github.andreyzebin.gitSql.pretty;

import io.github.andreyzebin.gitSql.sql.FilesIndex;
import java.sql.Connection;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.BiFunction;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class TablePrinter {

    private final TableComposition composition;
    private final String query;
    private final Map<String, BiFunction<Integer, Map<String, Optional<String>>, Integer>> aggregates;
    private final Map<String, AtomicInteger> totals = new HashMap<>();
    private TableStreamRow last;
    private Map<String, String> headers;
    private boolean headerPrinted = false;

    public TablePrinter(
            TableComposition composition,
            String query,
            Map<String, BiFunction<Integer, Map<String, Optional<String>>, Integer>> aggregates
    ) {
        this.composition = composition;
        this.query = query;
        this.aggregates = aggregates;
    }

    public static <K,V> Map<K, V> combine(
            Map<K, V> row,
            Map<K, V> collect
    ) {
        final HashMap<K, V> rowWithTotala = new HashMap<>(row);
        rowWithTotala.putAll(collect);
        return rowWithTotala;
    }

    public Stream<String> lines(
            Connection connection
    ) {
        final LinkedList<String> strings = new LinkedList<>();
        FilesIndex.streamRows(
                        (rs) -> row(FilesIndex.streamFields(rs)),
                        FilesIndex.query(connection, query)
                )
                .forEach(composition.getRowConsumer(strings));
        composition.getAfterRows(this, strings);

        return strings.stream();
    }

    public Map<String, String> getHeaders() {
        return headers;
    }

    public TableStreamRow row(Stream<Entry<String, String>> row) {
        final Map<String, Optional<String>> columns = row.collect(
                Collectors.toMap(Entry::getKey, v -> Optional.ofNullable(v.getValue())));
        if (headers == null) {
            headers = columns.entrySet().stream().collect(Collectors.toMap(Entry::getKey, Entry::getKey));
        }

        // update totals
        aggregates.keySet()
                .forEach(cVal -> {
                            final AtomicInteger metrica = totals.computeIfAbsent(cVal, (k) -> new AtomicInteger());
                            final BiFunction<Integer, Map<String, Optional<String>>, Integer> aggregator = aggregates.get(cVal);

                            metrica.set(aggregator.apply(metrica.get(), columns));
                        }
                );

        final TableStreamRow build = TableStreamRow.builder()
                .table(this)
                .row(columns)
                .totals(totals.entrySet()
                        .stream()
                        .collect(Collectors.toMap(Entry::getKey, ce -> ce.getValue().get())))
                .build();
        last = build;

        return build;

    }

    public boolean isHeaderPrinted() {
        return headerPrinted;
    }

    public void setHeaderPrinted(boolean headerPrinted) {
        this.headerPrinted = headerPrinted;
    }

    public TableStreamRow getLast() {
        return last;
    }
}
