package io.github.andreyzebin.gitSql.sql;

import io.github.andreyzebin.gitSql.FileSystem;
import java.io.Reader;
import java.nio.file.Path;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.AbstractMap.SimpleEntry;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.StringJoiner;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public class FilesIndex extends PersistedDb {

    private final FileSystem fileSystem;
    private final Map<Function<String, Boolean>, BiFunction<String, Reader, String>> mappers;

    public static final Map<Function<String, Boolean>, BiFunction<String, Reader, String>> JSON_MAPPERS =
            Map.of(
                    (f) -> f.endsWith(".json"),
                    (f, r) -> FileSystem.readLines(r).collect(Collectors.joining(System.lineSeparator()))
            );

    public FilesIndex(
            FileSystem fileSystem,
            Path storeDir,
            Map<Function<String, Boolean>, BiFunction<String, Reader, String>> mappers
    ) {
        super(storeDir);
        this.fileSystem = fileSystem;
        this.mappers = mappers;
    }

    public static FilesIndex of(
            FileSystem fileSystem,
            Path storeDir,
            Map<Function<String, Boolean>, BiFunction<String, Reader, String>> mappers) {
        return new FilesIndex(fileSystem, storeDir, mappers);
    }

    public static BiFunction<FileSystem, Path, FilesIndex> of(
            Map<Function<String, Boolean>, BiFunction<String, Reader, String>> mappers
    ) {
        return (a, b) -> new FilesIndex(a, b, mappers);
    }

    @Override
    protected void createSchema() throws SQLException {
        try (Statement dml = getConnection().createStatement();) {
            final String sql = """
                    CREATE TABLE IF NOT EXISTS files
                    (
                        ID      INT          auto_increment,
                        path    VARCHAR(256) NOT NULL,
                        data    JSON         NOT NULL,
                        UNIQUE NULLS ALL DISTINCT (path)
                    );""";

            log.debug(CommitsIndex.renderSqlLog(sql));
            dml.execute(sql);
        }
        super.createSchema();
    }

    @Override
    protected void reindex() {
        try {
            fileSystem.find(
                    (p) -> {
                        if (!fileSystem.isDir(p)) {
                            final String fileName = p.getFileName().toString();
                            mappers.entrySet()
                                    .stream()
                                    .filter(cKey -> cKey.getKey().apply(fileName))
                                    .findFirst()
                                    .ifPresent(
                                            cF -> merge(
                                                    getConnection(),
                                                    "files",
                                                    "path",
                                                    "data FORMAT JSON",
                                                    p.toString(),
                                                    cF.getValue().apply(fileName, fileSystem.get(p))
                                            )
                                    );
                        }
                        return true;
                    }
            );
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    public static <T> Stream<T> streamRows(Function<ResultSet, T> conv, ResultSet resultSet) {
        try {

            int count = 0;

            List<T> result = new LinkedList<>();

            while (resultSet.next()) {
                count++;
                result.add(conv.apply(resultSet));
            }

            return result.stream();

        } catch (SQLException e) {
            throw new RuntimeException(e);
        }
    }

    public static Stream<Entry<String, String>> streamFields(ResultSet rs) {
        try {
            ResultSetMetaData rsmd = rs.getMetaData();
            final int columnCount = rsmd.getColumnCount();
            List<Entry<String, String>> all = new LinkedList();
            for (int i = 1; i <= columnCount; i++) {
                String name = rsmd.getColumnName(i);
                all.add(new SimpleEntry<>(name, rs.getString(name)));
            }

            return all.stream();
        } catch (SQLException e) {
            throw new RuntimeException(e);
        }
    }

    public static int merge(Connection con, String tableAlias, String... values) {
        int length = (values.length) / 2;

        StringJoiner joiner = new StringJoiner(",");

        for (int i = 1; i < length; i++) {
            String s = String.format(" %s = incoming.%s", escapeFormat(values[i]), escapeFormat(values[i]));
            joiner.add(s);
        }

        List<String> wds = new LinkedList<>();
        for (int i = 0; i < length; i++) {
            wds.add(values[i].contains("FORMAT JSON") ? "? FORMAT JSON" : "?");
        }

        final String sql = """
                MERGE    into $table
                USING    values ($values) as incoming($fields)
                ON       $table.$conflictField = incoming.$conflictField
                WHEN not matched then
                INSERT
                    ($fields)
                    values
                    ($Inc.fields)
                WHEN     matched then
                UPDATE
                SET
                    $sets;
                """
                // set data = incoming.data
                .replace("$table", tableAlias)
                .replace(
                        "$values",
                        String.join(",", wds)
                )
                .replace(
                        "$fields",
                        Arrays.stream(values)
                                .sequential()
                                .map(FilesIndex::escapeFormat)
                                .limit(length)
                                .collect(Collectors.joining(","))
                )
                .replace(
                        "$Inc.fields",
                        Arrays.stream(values)
                                .sequential()
                                .map(FilesIndex::escapeFormat)
                                .limit(length)
                                .map(cF -> "incoming." + cF)
                                .collect(Collectors.joining(","))
                )
                .replace("$conflictField", escapeFormat(values[0]))
                .replace("$sets", joiner.toString());

        try (PreparedStatement statement = con.prepareStatement(sql);) {
            StringBuilder s = new StringBuilder(sql);
            // values ($values)
            for (int i = 0; i < length; i++) {
                final int parameterIndex = i + 1;
                final String value = escapeFormat(values[i + length]);
                statement.setString(parameterIndex, value);
                s
                        .append(parameterIndex)
                        .append(" = ")
                        .append(value).append(System.lineSeparator());
            }
            log.debug(CommitsIndex.renderSqlLog(s.toString()));
            return statement.executeUpdate();
        } catch (SQLException e) {
            throw new RuntimeException(e);
        }
    }

    private static String escapeFormat(String values) {
        return values.replace(" FORMAT JSON", "");
    }

    public static ResultSet query(Connection con, String sql, String... values) {
        PreparedStatement statement = null;
        try {
            statement = con.prepareStatement(sql);

            log.debug(CommitsIndex.renderSqlLog(sql));

            for (int i = 0; i < values.length; i++) {
                statement.setString(i + 1, values[i]);
            }

            return statement.executeQuery();
        } catch (SQLException e) {
            throw new RuntimeException(e);
        }

    }

    public static int queryUpdate(Connection con, String sql, String... values) {
        try (PreparedStatement statement = con.prepareStatement(sql);) {
            log.debug(CommitsIndex.renderSqlLog(sql));

            for (int i = 0; i < values.length; i++) {
                statement.setString(i + 1, values[i]);
            }

            return statement.executeUpdate();
        } catch (SQLException e) {
            throw new RuntimeException(e);
        }
    }

}
