package io.github.andreyzebin.gitSql.sql;

import io.github.andreyzebin.gitSql.git.GitFS;
import io.github.andreyzebin.gitSql.git.VersionControl.Commit;
import java.nio.file.Path;
import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Statement;
import java.time.Instant;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public class TimeSeriesQuery extends PersistedDb {

    private static final String COLUMN_NAME_POINT = "ts_point";
    private static final String COLUMN_NAME_HASH = "ts_hash";
    private static final String TABLE_NAME_RESULT = "result";
    private final GitFS dataSet;
    private final JdbcView dataSetView;
    private final Instant from;
    private final Instant to;
    private final String sql;
    private final List<String> resultAliases = new LinkedList<>();

    public TimeSeriesQuery(
            String sql,
            GitFS dataSet,
            Instant from,
            Instant to,
            BiFunction<? super GitFS, Path, ? extends JdbcView> indexerFactory
    ) {
        super(Path.of("run", "tsq_result"));
        this.dataSet = dataSet;
        this.dataSetView = indexerFactory.apply(dataSet, Path.of("run", "tsq_dataset"));
        this.from = from;
        this.to = to;
        this.sql = sql;
    }

    @Override
    protected void createSchema() throws SQLException {
        // schema is only known after some point within reindex()
    }

    @Override
    protected void reindex() {
        Connection connection = getConnection();
        // run query
        final Iterator<Commit> commitsAll = dataSet.commits().collect(Collectors.toList()).reversed().iterator();

        final List<Instant> points = rangeSplit(from, to).toList();
        Commit cCommitPrev = commitsAll.next();
        Commit cCommitAfter = commitsAll.next();

        Set<Commit> commits = new HashSet<>();

        for (Instant cPoint : points) {
            // move till point
            // find closest commits

            long distPrev = getMilliDistance(cCommitPrev.getTimestampInstant(), cPoint);
            long distAfter = getMilliDistance(cCommitAfter.getTimestampInstant(), cPoint);

            while (distPrev > distAfter && commitsAll.hasNext()) {
                // move window
                cCommitPrev = cCommitAfter;
                cCommitAfter = commitsAll.next();

                distPrev = getMilliDistance(cCommitPrev.getTimestampInstant(), cPoint);
                distAfter = getMilliDistance(cCommitAfter.getTimestampInstant(), cPoint);
            }
            // distPrev is best choose

            final Commit bestHit = distPrev < distAfter ? cCommitPrev : cCommitAfter;
            if (commits.add(bestHit)) {
                pullCommit(bestHit, connection);
            }
        }

    }

    private void pullCommit(Commit cCommitPrev, Connection connection) {
        dataSet.seek(cCommitPrev.getHash());
        dataSetView.drop();

        FilesIndex.streamRows(
                (rs) -> {
                    boolean needFillHeader = resultAliases.isEmpty();

                    if (needFillHeader) {
                        resultAliases.add(COLUMN_NAME_POINT);
                        resultAliases.add(COLUMN_NAME_HASH);
                    }

                    List<String> row = new LinkedList<>();
                    row.add(cCommitPrev.getTimestampInstant().toString());
                    row.add(cCommitPrev.getHash());
                    FilesIndex.streamFields(rs)
                            .forEach(
                                    cColumn -> {
                                        if (needFillHeader) {
                                            resultAliases.add(cColumn.getKey());
                                        }

                                        row.add(cColumn.getValue());
                                    }
                            );

                    LinkedList<String> rov = new LinkedList<>(resultAliases);
                    rov.addAll(row);
                    return rov;
                },
                FilesIndex.query(dataSetView.getConnection(), sql)
        ).forEach(
                cRow -> {
                    createSchema2();
                    FilesIndex.merge(connection, TABLE_NAME_RESULT, cRow.toArray(String[]::new));
                }
        );
    }


    private void createSchema2() {
        if (schemaCreated) {
            return;
        }
        if (storeExists()) {
            try (Statement dml = getConnection().createStatement();) {
                StringBuilder sql1 = new StringBuilder(
                        String.format("""
                                        CREATE TABLE IF NOT EXISTS %s
                                        (
                                        ts_ID INT auto_increment,
                                        """,
                                TABLE_NAME_RESULT
                        )
                );

                for (int i = 0; i < resultAliases.size(); i++) {
                    sql1.append(resultAliases.get(i)).append(" VARCHAR(256) ");
                    if (i + 1 < resultAliases.size()) {
                        sql1.append(", \n");
                    } else {
                        sql1.append(" \n");
                    }
                }

                sql1.append(");");

                log.debug(CommitsIndex.renderSqlLog(sql1.toString()));

                dml.execute(sql1.toString());

                super.createSchema();
            } catch (SQLException e) {
                throw new RuntimeException(e);
            }
        }
    }

    public static Stream<Instant> rangeSplit(Instant from, Instant to) {
        final long milliDelta = getMilliDelta(from, to);
        final int TIME_PIECES = 10;
        long step = milliDelta / TIME_PIECES;

        return Stream.iterate(from, (i) -> i.plusMillis(step)).limit(TIME_PIECES + 1);
    }

    private static long getMilliDelta(Instant start, Instant finish) {
        return finish.toEpochMilli() - start.toEpochMilli();
    }

    private static long getMilliDistance(Instant a, Instant b) {
        return Math.abs(getMilliDelta(a, b));
    }
}
