package io.github.andreyzebin.gitSql.git2;

import io.github.andreyzebin.gitSql.bash.BashCSS;
import io.github.andreyzebin.gitSql.bash.BashTML;
import io.github.andreyzebin.gitSql.bash.Color;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.function.Consumer;
import java.util.function.Function;
import lombok.Builder;
import lombok.Data;
import lombok.With;
import lombok.extern.slf4j.Slf4j;
import org.slf4j.MDC;

@Slf4j
public class TerminalFun implements TerminalIO {

    private final TerminalIO delegate;
    private final String rand = UUID.randomUUID().toString().substring(0, 4);
    private final TerminalColors css;


    public TerminalFun(TerminalIO delegate, TerminalColors css) {
        this.delegate = delegate;
        this.css = css;
    }

    public TerminalFun(TerminalIO delegate) {
        this.delegate = delegate;
        this.css = TerminalColors.DEFAULT;
    }

    public int exec(String comm, Consumer<String> stdout, Consumer<String> stderr) {
        return exec(comm, comm, stdout, stderr);
    }

    @Override
    public int exec(String comm, String mask, Consumer<String> stdout, Consumer<String> stderr) {
        try (
                LineWriter errLog = new LineWriter(s -> log.error(css.stderr.apply(s)));
                LineWriter outLog = new LineWriter(s -> log.debug(css.stdout.apply(s)));
                LineWriter cmdLog = new LineWriter(s -> log.debug(css.stdin.apply(s)));
        ) {

            String pwd = delegate.eval("pwd");
            String user = delegate.eval("whoami");
            MDC.put("terminal.user", css.user.apply(user));
            MDC.put("terminal.dir", css.dir.apply(pwd));
            MDC.put("terminal.id", css.id.apply(rand));
            cmdLog.println(css.cmd.apply(mask));
            return delegate.exec(
                    comm,
                    fork(stdout, outLog::print),
                    fork(stderr, errLog::print)
            );
        } catch (Exception e) {
            throw new RuntimeException(e);
        } finally {
            MDC.clear();
        }
    }


    public static <T> Consumer<T> fork(Consumer<T> left, Consumer<T> right) {
        return (ffu) -> {
            left.accept(ffu);
            right.accept(ffu);
        };
    }

    @Builder
    @Data
    @With
    public static class TerminalColors {

        public static final TerminalColors DEFAULT = getDefault();

        public static TerminalColors getDefault() {
            ColorPool defaults = ColorPool.defaults();
            return builder()
                    .stderr(s -> BashCSS.stdErrRender(new BashTML(s)).toString())
                    .stdout(s -> BashCSS.stdRender(new BashTML(s)).toString())
                    .stdin(s -> s)
                    .id(s -> new BashTML(s).fill(defaults.getColor(s)).toString())
                    .cmd(BashCSS::bashRender)
                    .user(s -> new BashTML(s).fill(Color.MAGENTA).toString())
                    .build();
        }

        @Builder.Default
        private Function<String, String> stderr = s -> s;
        @Builder.Default
        private Function<String, String> stdout = s -> s;
        @Builder.Default
        private Function<String, String> stdin = s -> s;
        @Builder.Default
        private Function<String, String> user = s -> s;
        @Builder.Default
        private Function<String, String> dir = s -> s;
        @Builder.Default
        private Function<String, String> id = s -> s;
        @Builder.Default
        private Function<String, String> cmd = s -> s;
    }


    public static class ColorPool {

        private final Set<Color> pool;
        private final Map<String, Color> colors = new HashMap<>();


        public ColorPool(Set<Color> pool) {
            this.pool = pool;
        }

        public static ColorPool defaults() {
            Set<Color> pool = new HashSet<>();
            pool.add(Color.BLUE);
            pool.add(Color.GREEN);
            pool.add(Color.CYAN);
            pool.add(Color.MAGENTA);
            pool.add(Color.RED);

            return new ColorPool(pool);
        }


        public Color getColor(String rand) {
            return colors.computeIfAbsent(rand, (kk) -> {
                Iterator<Color> iterator = pool.iterator();
                if (iterator.hasNext()) {
                    Color next = iterator.next();
                    iterator.remove();
                    return next;
                } else {
                    return Color.WHITE_BRIGHT;
                }
            });
        }
    }
}