package io.github.andreyzebin.gitSql.git2;

import io.github.andreyzebin.gitSql.bash.Bash;
import io.github.andreyzebin.gitSql.git.GitAuth;
import io.github.andreyzebin.gitSql.git.GitSource;
import io.github.andreyzebin.gitSql.git.BranchHead;
import io.github.andreyzebin.gitSql.git.GitVersions;
import java.net.URI;
import java.net.URLEncoder;
import java.nio.file.Path;
import java.util.Objects;
import java.util.UUID;
import java.util.stream.Stream;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public class RemoteOrigin extends AbstractVersionControl implements GitSource, GitVersions {

    protected String gitUri;
    private final TerminalIO bash;
    private final GitAuth authStrategy;
    protected Path root;
    private final String branch;
    private final String home;


    public RemoteOrigin(String gitUri, TerminalIO bash, GitAuth authStrategy) {
        this(gitUri, bash, authStrategy, null);
    }

    public RemoteOrigin(String gitUri, TerminalIO bash, GitAuth authStrategy, String branch) {
        super(bash);
        this.branch = branch;
        this.gitUri = gitUri;
        this.bash = bash;
        this.authStrategy = authStrategy;
        this.home = bash.eval("pwd");
    }

    @Override
    public void close() throws Exception {
        if (root != null) {
            String pwd = bash.eval("pwd");
            try {
                log.debug("Cleaning up temporary git root...");
                bash.eval("cd " + Bash.escape(Bash.posix(root.getParent())));
                bash.eval("rm -rf " + root.getFileName().toString());
            } finally {
                bash.eval("cd " + pwd);
            }
        }
    }

    public Stream<BranchHead> listBranches() {
        authStrategy.beforeRemote(bash);
        Stream<BranchHead> remoteBranches = GitBindings.getRemoteBranches(
                authStrategy.injectAuth(gitUri),
                bash,
                authStrategy.useAuthHeader()
        );
        authStrategy.afterRemote(bash);

        return remoteBranches;
    }

    @Override
    public Path getRoot() {
        if (root == null) {
            log.debug("Initializing Git root...");
            String pwd = bash.eval("pwd");
            String tempDirName = getRandDir();

            try {
                bash.eval("cd " + Bash.escape(home));
                bash.eval("mkdir -p " + tempDirName);
                bash.eval("cd " + tempDirName);
                authStrategy.beforeRemote(bash);
                // clone alternatives

                // git init .
                // git remote add origin <repository-url>
                // git pull origin master

                // git init .
                // git remote add -f origin <repository-url>
                // git checkout master

                // git remote set-head origin -a
                GitBindings.clone(
                        ".",
                        bash,
                        authStrategy.injectAuth(gitUri),
                        false,
                        authStrategy.useAuthHeader(),
                        branch
                );
                authStrategy.afterRemote(bash);

                return root = Path.of(home).resolve(tempDirName);
            } catch (Exception e) {
                log.error("Error while loading from remote: ", e);
                throw new RuntimeException(e);
            } finally {
                bash.eval("cd " + pwd);
            }
        }
        return root;
    }

    @Override
    public void flush() {
        if (root != null) {
            String pwd = bash.eval("pwd");
            try {
                bash.eval("cd " + Bash.escape(Bash.posix(getRoot())));
                authStrategy.beforeRemote(bash);
                GitBindings.push(bash);
                authStrategy.afterRemote(bash);
            } finally {
                bash.eval("cd " + pwd);
            }
        }
    }

    @Override
    public void pull() {
        String pwd = bash.eval("pwd");
        try {
            bash.eval("cd " + Bash.escape(Bash.posix(getRoot())));
            authStrategy.beforeRemote(bash);
            GitBindings.pull(bash);
            authStrategy.afterRemote(bash);
        } finally {
            bash.eval("cd " + pwd);
        }
    }

    private String getRandDir() {
        String rand = UUID.randomUUID().toString().substring(0, 4);
        String[] split = gitUri.replace(":", "://")
                .replace("@", URLEncoder.encode("@"))
                .replace(".git", "")
                .split("/");
        String name = split[split.length - 1];
        return name + "_" + rand;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        RemoteOrigin that = (RemoteOrigin) o;
        return Objects.equals(gitUri, that.gitUri) && Objects.equals(branch, that.branch);
    }

    @Override
    public int hashCode() {
        return Objects.hash(gitUri, branch);
    }
}
