package io.github.andreyzebin.gitSql.sql;

import io.github.andreyzebin.gitSql.FileSystem;
import io.github.andreyzebin.gitSql.git.VersionControl;
import io.github.andreyzebin.gitSql.git.VersionControl.Change;
import io.github.andreyzebin.gitSql.git2.VersionedFiles;
import io.github.andreyzebin.gitSql.sql.SqlUtils.SchemaBuilder;
import java.io.Reader;
import java.nio.file.Path;
import java.sql.SQLException;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public class FilesIndexMerger implements TableSource {

    private final VersionedFiles fileSystem;
    private final String alias;
    private final Map<Function<String, Boolean>, BiFunction<String, Reader, String>> mappers;

    private String updates = null;

    private static final String JSON = ".json";

    public interface FunctionCode<A, B> extends Function<A, B> {

        String getCode();

    }

    public static <A, B> Function<A, B> ofCode(String code, Function<A, B> fu) {
        return new FunctionCode<>() {

            @Override
            public String getCode() {
                return code;
            }

            @Override
            public B apply(A f) {
                return fu.apply(f);
            }

            @Override
            public boolean equals(Object obj) {
                if (!(obj instanceof FunctionCode)) {
                    return false;
                }
                return Objects.equals(((FunctionCode<?, ?>) obj).getCode(), this.getCode());
            }

            @Override
            public int hashCode() {
                return Objects.hash(getCode());
            }
        };
    }

    public static <A, B, C> BiFunction<A, B, C> ofCode(String code, BiFunction<A, B, C> fu) {
        return new BiFunctionCode<>() {

            @Override
            public C apply(A a, B b) {
                return fu.apply(a, b);
            }

            @Override
            public String getCode() {
                return code;
            }

            @Override
            public boolean equals(Object obj) {
                if (!(obj instanceof BiFunctionCode)) {
                    return false;
                }
                return Objects.equals(((BiFunctionCode<?, ?, ?>) obj).getCode(), this.getCode());
            }

            @Override
            public int hashCode() {
                return Objects.hash(getCode());
            }
        };
    }

    public interface BiFunctionCode<A, B, C> extends BiFunction<A, B, C> {

        String getCode();
    }

    public static final Map<Function<String, Boolean>, BiFunction<String, Reader, String>> JSON_MAPPERS =
            Map.of(
                    ofCode(JSON, f -> f.endsWith(JSON)),
                    ofCode(JSON, (f, r) -> FileSystem.readLines(r).collect(Collectors.joining(System.lineSeparator())))
            );

    public FilesIndexMerger(
            VersionedFiles fileSystem,
            String alias,
            Map<Function<String, Boolean>, BiFunction<String, Reader, String>> mappers
    ) {
        this.fileSystem = fileSystem;
        this.alias = alias;
        this.mappers = mappers;
    }

    @Override
    public TableSource updates(String fromVersion) {
        updates = fromVersion;
        return this;
        // return TableSource.super.updates(fromVersion);
    }

    @Override
    public String version() {
        return getHash();
        // return TableSource.super.version();
    }

    private String getHash() {
        return fileSystem.getSource().commits().findFirst().get().getHash();
    }

    @Override
    public SchemaBuilder createSchema() throws SQLException {
        return new SchemaBuilder(alias)
                .withColumn("path")
                .withColumn("data", "data JSON NOT NULL")
                .withColumn("origin")
                .withMerge("origin", "path")
                .withIndex("UNIQUE NULLS ALL DISTINCT (origin, path)");
    }

    @Override
    public Stream<Map<String, String>> rows() {
        List<Map<String, String>> rows = new LinkedList<>();

        if (updates == null) {
            fileSystem.find(
                    (p) -> {
                        if (!fileSystem.isDir(p)) {
                            addRow(p, rows);
                        }
                        return true;
                    }
            );
        } else {
            fileSystem.getSource().changes(updates)
                    .filter(Change::isUpdate)
                    .forEach(
                            change -> {
                                addRow(Path.of(change.getFile()), rows);
                            }
                    );
        }

        return rows.stream();
    }

    private void addRow(Path p, List<Map<String, String>> rows) {
        String fileName = p.getFileName().toString();
        mappers.entrySet()
                .stream()
                .filter(cKey -> cKey.getKey().apply(fileName))
                .findFirst()
                .ifPresent(
                        cF -> {
                            rows.add(Map.of(
                                    "path",
                                    p.toString(),
                                    "data FORMAT JSON",
                                    cF.getValue().apply(fileName, fileSystem.get(p)),
                                    "origin",
                                    fileSystem.getSource().origin().orElse(fileSystem.getRoot().toString()))
                            );
                        }
                );
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        FilesIndexMerger that = (FilesIndexMerger) o;
        return Objects.equals(fileSystem.getSource(), that.fileSystem.getSource())
                && Objects.equals(alias, that.alias)
                && Objects.equals(mappers, that.mappers);
    }

    @Override
    public int hashCode() {
        return Objects.hash(fileSystem.getSource(), alias, mappers, this.getClass().getName());
    }
}