package io.github.andreyzebin.gitSql.sql;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.util.AbstractMap.SimpleEntry;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.StringJoiner;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import lombok.Builder;
import lombok.Data;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public class SqlUtils {

    public static <T> Stream<T> streamRows(Function<ResultSet, T> conv, ResultSet resultSet) {
        try {
            List<T> result = new LinkedList<>();

            while (resultSet.next()) {
                result.add(conv.apply(resultSet));
            }

            return result.stream();
        } catch (SQLException e) {
            throw new RuntimeException(e);
        }
    }

    public static Stream<Entry<String, String>> streamFields(ResultSet rs) {
        try {
            ResultSetMetaData rsmd = rs.getMetaData();
            final int columnCount = rsmd.getColumnCount();
            List<Entry<String, String>> all = new LinkedList<>();
            for (int i = 1; i <= columnCount; i++) {
                String name = rsmd.getColumnName(i);
                all.add(new SimpleEntry<>(name, rs.getString(name)));
            }

            return all.stream();
        } catch (SQLException e) {
            throw new RuntimeException(e);
        }
    }

    public static int merge(Connection con, String tableAlias, String... values) {
        int length = (values.length) / 2;

        StringJoiner joiner = new StringJoiner(",");

        for (int i = 1; i < length; i++) {
            String s = String.format(" %s = incoming.%s", SqlUtils.escapeFormat(values[i]),
                    SqlUtils.escapeFormat(values[i]));
            joiner.add(s);
        }

        List<String> wds = new LinkedList<>();
        for (int i = 0; i < length; i++) {
            wds.add(values[i].contains("FORMAT JSON") ? "? FORMAT JSON" : "?");
        }

        final String sql = """
                MERGE    into $table
                USING    values ($values) as incoming($fields)
                ON       $table.$conflictField = incoming.$conflictField
                WHEN not matched then
                INSERT
                    ($fields)
                    values
                    ($Inc.fields)
                WHEN     matched then
                UPDATE
                SET
                    $sets;
                """
                // set data = incoming.data
                .replace("$table", tableAlias)
                .replace(
                        "$values",
                        String.join(",", wds)
                )
                .replace(
                        "$fields",
                        Arrays.stream(values)
                                .sequential()
                                .map(SqlUtils::escapeFormat)
                                .limit(length)
                                .collect(Collectors.joining(","))
                )
                .replace(
                        "$Inc.fields",
                        Arrays.stream(values)
                                .sequential()
                                .map(SqlUtils::escapeFormat)
                                .limit(length)
                                .map(cF -> "incoming." + cF)
                                .collect(Collectors.joining(","))
                )
                .replace("$conflictField", SqlUtils.escapeFormat(values[0]))
                .replace("$sets", joiner.toString());

        try (PreparedStatement statement = con.prepareStatement(sql);) {
            StringBuilder s = new StringBuilder(sql);
            for (int i = 0; i < length; i++) {
                final int parameterIndex = i + 1;
                final String value = SqlUtils.escapeFormat(values[i + length]);
                statement.setString(parameterIndex, value);
                s
                        .append(parameterIndex)
                        .append(" = ")
                        .append(value).append(System.lineSeparator());
            }
            log.debug(CommitsIndex.renderSqlLog(s.toString()));
            return statement.executeUpdate();
        } catch (SQLException e) {
            throw new RuntimeException(e);
        }
    }

    public static int merge(
            Connection con,
            String tableAlias,
            List<String> mergeColumns,
            Map<String, String> values
    ) {
        int length = values.size();

        StringJoiner joiner = new StringJoiner(",");
        List<String> columns = values.keySet().stream().toList();

        for (int i = 0; i < length; i++) {
            if (
                    !mergeColumns.contains(SqlUtils.escapeFormat(columns.get(i))) &&
                            !mergeColumns.contains(columns.get(i))
            ) {
                String s = String.format(" %s = incoming.%s", SqlUtils.escapeFormat(columns.get(i)),
                        SqlUtils.escapeFormat(columns.get(i)));
                joiner.add(s);
            }
        }

        List<String> wds = new LinkedList<>();
        for (int i = 0; i < length; i++) {
            wds.add(columns.get(i).contains("FORMAT JSON") ? "? FORMAT JSON" : "?");
        }

        // $table.$conflictField = incoming.$conflictField
        final String sql = """
                MERGE    into $table
                USING    values ($values) as incoming($fields)
                ON       $conflicts
                WHEN not matched then
                INSERT
                    ($fields)
                    values
                    ($Inc.fields)
                WHEN     matched then
                UPDATE
                SET
                    $sets;
                """
                // set data = incoming.data
                .replace("$table", tableAlias)
                .replace(
                        "$values",
                        String.join(",", wds)
                )
                .replace(
                        "$fields",
                        columns.stream()
                                .map(SqlUtils::escapeFormat)
                                .collect(Collectors.joining(","))
                )
                .replace(
                        "$Inc.fields",
                        columns.stream()
                                .map(SqlUtils::escapeFormat)
                                .map(cF -> "incoming." + cF)
                                .collect(Collectors.joining(","))
                )
                .replace("$conflictField", SqlUtils.escapeFormat(mergeColumns.get(0)))
                .replace("$conflicts", mergeColumns.stream()
                        .map(col -> "$table.$conflictField = incoming.$conflictField"
                                .replace("$table", tableAlias)
                                .replace("$conflictField", col)

                        ).collect(Collectors.joining(" AND ")))
                .replace("$sets", joiner.toString());

        try (PreparedStatement statement = con.prepareStatement(sql);) {
            StringBuilder s = new StringBuilder(sql);
            for (int i = 0; i < length; i++) {
                final int parameterIndex = i + 1;
                final String value = SqlUtils.escapeFormat(values.get(columns.get(i)));
                statement.setString(parameterIndex, value);
                s
                        .append(parameterIndex)
                        .append(" = ")
                        .append(value).append(System.lineSeparator());
            }
            log.debug(CommitsIndex.renderSqlLog(s.toString()));
            return statement.executeUpdate();
        } catch (SQLException e) {
            throw new RuntimeException(e);
        }
    }

    private static String escapeFormat(String values) {
        return values.replace(" FORMAT JSON", "");
    }

    public static ResultSet query(Connection con, String sql, String... values) {
        PreparedStatement statement = null;
        try {
            statement = con.prepareStatement(sql);

            log.debug(CommitsIndex.renderSqlLog(sql));

            for (int i = 0; i < values.length; i++) {
                statement.setString(i + 1, values[i]);
            }

            return statement.executeQuery();
        } catch (SQLException e) {
            throw new RuntimeException(e);
        }

    }

    public static int queryUpdate(Connection con, String sql, String... values) {
        try (PreparedStatement statement = con.prepareStatement(sql);) {
            log.debug(CommitsIndex.renderSqlLog(sql));

            for (int i = 0; i < values.length; i++) {
                statement.setString(i + 1, values[i]);
            }

            return statement.executeUpdate();
        } catch (SQLException e) {
            throw new RuntimeException(e);
        }
    }

    @Data
    @Builder
    public static class FieldSource {

        private final String alias;
    }

    @Getter
    public static class SchemaBuilder {

        private final String alias;
        private final Map<String, Optional<String>> columns;
        private final List<String> indexes;
        private final List<String> mergeColumns;

        public SchemaBuilder(String alias) {
            this.alias = alias;
            indexes = new ArrayList<>();
            columns = new HashMap<>();
            mergeColumns = new LinkedList<>();
        }

        public SchemaBuilder withColumn(String columnAlias) {
            columns.put(columnAlias, Optional.empty());
            return this;
        }

        public SchemaBuilder withColumn(String columnAlias, String overr) {
            columns.put(columnAlias, Optional.of(overr));
            return this;
        }

        public SchemaBuilder withIndex(String index) {
            indexes.add(index);
            return this;
        }

        public SchemaBuilder withMerge(String... columnAliases) {
            mergeColumns.addAll(List.of(columnAliases));
            return this;
        }

        public String toDML() {
            StringBuilder sql1 = new StringBuilder(
                    String.format("""
                                    CREATE TABLE IF NOT EXISTS %s
                                    (
                                    ID INT auto_increment""",
                            alias
                    )
            );

            columns.forEach(
                    (k, v) -> {
                        sql1.append(",\n");
                        if (v.isPresent()) {
                            sql1.append(v.get());
                        } else {
                            sql1.append(k);
                            sql1.append(" VARCHAR(512)");
                        }
                    }
            );

            indexes.forEach(cIndex -> {
                sql1.append(",\n");
                sql1.append(cIndex);
            });

            sql1.append(");");
            return sql1.toString();
        }
    }

    public static StringBuilder renderSchema(String tableName, List<String> aliases) {
        StringBuilder sql1 = new StringBuilder(
                String.format("""
                                CREATE TABLE IF NOT EXISTS %s
                                (
                                ts_ID INT auto_increment,
                                """,
                        tableName
                )
        );

        for (int i = 0; i < aliases.size(); i++) {
            sql1.append(aliases.get(i)).append(" VARCHAR(256) ");
            if (i + 1 < aliases.size()) {
                sql1.append(", \n");
            } else {
                sql1.append(" \n");
            }
        }

        sql1.append(");");
        return sql1;
    }
}
