package io.github.andreyzebin.gitSql.git2;

import io.github.andreyzebin.gitSql.bash.Bash;
import io.github.andreyzebin.gitSql.git.VersionControl;
import java.nio.file.Path;
import java.time.Instant;
import java.util.Optional;
import java.util.stream.Stream;

public abstract class AbstractVersionControl implements VersionControl {

    protected final TerminalIO term;
    private Optional<String> origin = null;

    protected AbstractVersionControl(TerminalIO term) {
        this.term = term;
    }

    /**
     * Fast-forward
     */
    @Override
    public VersionControl seek(String commit) {
        String pwd = term.eval("pwd");
        try {
            term.eval("cd " + Bash.encode(getRoot()));
            GitBindings.checkout(commit, term);
            return this;
        } finally {
            term.eval("cd " + pwd);
        }
    }

    /**
     * Fast-forward
     */
    @Override
    public Stream<Commit> commits() {
        String pwd = term.eval("pwd");
        try {
            term.eval("cd " + Bash.encode(getRoot()));
            return GitBindings.commitsList(term);
        } finally {
            term.eval("cd " + pwd);
        }
    }

    @Override
    public Stream<? extends Change> changes(String hashFrom) {
        String pwd = term.eval("pwd");
        try {
            term.eval("cd " + Bash.encode(getRoot()));
            return GitBindings.filesChangedQuery(
                    GitBindings.ALL_BRANCHES,
                    GitBindings.sinceHash(hashFrom),
                    term
            );
        } finally {
            term.eval("cd " + pwd);
        }
    }

    @Override
    public Instant timestamp() {
        String pwd = term.eval("pwd");
        try {
            term.eval("cd " + Bash.encode(getRoot()));
            return GitBindings.commitsList(term)
                    .findFirst()
                    .get()
                    .getTimestampInstant();
        } finally {
            term.eval("cd " + pwd);
        }
    }

    public Optional<String> origin() {
        String pwd = term.eval("pwd");
        try {
            if (origin == null) {
                term.eval("cd " + Bash.encode(getRoot()));
                origin = GitBindings.getOrigin(term);
            }
            return origin;
        } finally {
            term.eval("cd " + pwd);
        }
    }

    public Optional<String> branch() {
        String pwd = term.eval("pwd");
        try {
            term.eval("cd " + Bash.encode(getRoot()));
            return GitBindings.getBranch(term);
        } finally {
            term.eval("cd " + pwd);
        }
    }

    abstract public Path getRoot();
}
