package io.github.andreyzebin.gitSql.sql;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public abstract class PersistedIndex implements JdbcView {

    private final Path storeDir;
    private Connection connection;
    protected Path lastStore;
    protected boolean schemaCreated = false;

    public PersistedIndex(Path storeDir) {
        this.storeDir = storeDir.toAbsolutePath();
    }

    @Override
    public Connection getConnection() {
        try {
            if (connection != null && !connection.isClosed()) {
                return connection;
            }
            log.info("Initializing DB store: {}", storeDir);
            Files.createDirectories(storeDir);
            final boolean justCreated = !storeExists();
            String jdbcURL = "jdbc:h2:" + storeDir.resolve("data");
            connection = DriverManager.getConnection(jdbcURL, "", "");
            log.info("Connected to H2 embedded database.");

            if (justCreated) {
                createSchema();
                reindex();
            } else {
                init();
            }

            return connection;

        } catch (SQLException | IOException e) {
            throw new RuntimeException(e);
        }
    }

    public boolean storeExists() {
        lastStore = lastStore == null ? storeDir.resolve("data.mv.db") : lastStore;
        return Files.exists(lastStore);
    }

    protected void createSchema() throws SQLException {
        if (storeExists()) {
            schemaCreated = true;
        }
    }

    protected void init() {

    };

    protected void reindex() {

    };

    @Override
    public PersistedIndex drop() {
        try {
            connection = null;
            if (storeExists()) {
                Files.deleteIfExists(lastStore);
                schemaCreated = false;
                log.info("Deleted DB store file: {}", lastStore);
            }
        } catch (Exception e) {
            throw new RuntimeException(e);
        }

        return this;
    }

    @Override
    public void close() throws Exception {
        if (connection != null) {
            connection.close();
        }
    }
}
