package io.github.andreyzebin.gitSql;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import io.github.zebin.javabash.sandbox.PosixPath;
import org.yaml.snakeyaml.DumperOptions;
import org.yaml.snakeyaml.LoaderOptions;
import org.yaml.snakeyaml.Yaml;
import org.yaml.snakeyaml.constructor.Constructor;
import org.yaml.snakeyaml.introspector.Property;
import org.yaml.snakeyaml.nodes.NodeTuple;
import org.yaml.snakeyaml.nodes.Tag;
import org.yaml.snakeyaml.representer.Representer;

import java.io.*;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Comparator;
import java.util.List;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public interface FileSystemUtils extends AutoCloseable {

    static <T> T loadYaml(Reader strings, Class<T> clazz) {
        LoaderOptions lo = new LoaderOptions();
        lo.setTagInspector(tag -> true);
        Representer r = new Representer(new DumperOptions());
        r.getPropertyUtils().setSkipMissingProperties(true);

        Yaml yaml = new Yaml(new Constructor(clazz, lo), r);

        final String contents = loadFile(strings);
        return yaml.load(contents.isBlank() ? "f: null" : contents);
    }

    private static String loadFile(Reader strings) {
        try (BufferedReader br = new BufferedReader(strings)) {
            return br.lines().collect(Collectors.joining(System.lineSeparator()));
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private static Representer getIgnoreNullsRepresenter(DumperOptions dumperOptions) {
        return new Representer(dumperOptions) {
            @Override
            protected NodeTuple representJavaBeanProperty(Object javaBean, Property property, Object propertyValue,
                    Tag customTag) {
                // if value of property is null, ignore it.
                if (propertyValue == null) {
                    return null;
                } else {
                    return super.representJavaBeanProperty(javaBean, property, propertyValue, customTag);
                }
            }
        };
    }

    static <T> void dumpYaml(T meta, Writer dest) {
        DumperOptions op = new DumperOptions();
        Yaml yaml = new Yaml(getIgnoreNullsRepresenter(op));

        try (BufferedWriter bw = new BufferedWriter(dest)) {
            yaml.dump(meta, bw);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    static <T> T readJson(Reader r, TypeReference<T> tr) {
        String json = loadFile(r);
        ObjectMapper mapper = new ObjectMapper()
                .configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);

        T java = null;

        try {
            return java = mapper.readValue(json, tr);
        } catch (JsonMappingException e) {
            throw new RuntimeException(e);
        } catch (JsonProcessingException e) {
            throw new RuntimeException(e);
        }
    }

    public static Stream<String> readLines(Reader r) {
        return loadFile(r).lines();
    }

    static <T> Reader toJson(T value) {

        ObjectMapper mapper = new ObjectMapper()
                .configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);

        String java = null;

        try {
            java = mapper.writeValueAsString(value);
            return new InputStreamReader(new ByteArrayInputStream(java.getBytes(StandardCharsets.UTF_8)));
        } catch (JsonProcessingException e) {
            throw new RuntimeException(e);
        }
    }

    public static void pipe(Reader source, Writer dest) {
        try (BufferedWriter bw = new BufferedWriter(dest);
                BufferedReader br = new BufferedReader(source);) {
            br.lines()
                    .forEach(
                            cLine -> {
                                try {

                                    bw.write(cLine);
                                    bw.newLine();
                                } catch (IOException e) {
                                    throw new RuntimeException(e);
                                }
                            }
                    );

        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    static void deleteRecursively(Path root) {
        try (Stream<Path> walk = Files.walk(root)) {
            walk.sorted(Comparator.reverseOrder())
                    .map(Path::toFile)
                    .forEach(File::delete);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    static Comparator<Path> filesFirst() {
        return (o1, o2) -> {
            int f1 = Files.isRegularFile(o1) ? 1 : 0;
            int f2 = Files.isRegularFile(o2) ? 1 : 0;
            return f2 - f1;
        };
    }

    static void tVerse(
            Path dir,
            Function<Path, Boolean> needContinue,
            Consumer<Path> sayBye,
            Comparator<Path> sorting

    ) {
        if (Files.isDirectory(dir)) {
            try {
                List<Path> sorted;
                try (Stream<Path> fList = Files.list(dir)) {
                    sorted = fList.sorted(sorting)
                            .toList();
                }

                for (Path cFile : sorted) {
                    Boolean isContinue = needContinue.apply(cFile);
                    if (Files.isDirectory(cFile) && isContinue) {
                        tVerse(cFile, needContinue, sayBye, sorting);
                        sayBye.accept(cFile);
                    } else if (!isContinue) {
                        break;
                    }
                }

            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
    }
}
