package io.github.andreyzebin.gitSql.config;

import io.github.andreyzebin.gitSql.git.GitAPI;
import io.github.zebin.javabash.sandbox.PosixPath;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;

import java.time.Instant;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class ConfigVersions implements ConfigHistory, AutoCloseable {

    public static final String IO_GITHUB_VEZUVIO = "io.github.vezuvio";
    public static final long DEFAULT_LOCK_TIMEOUT_SEC = 60 * 60; // 1 hour
    private final GitAPI vc;
    private final ConfigTree ct;

    public ConfigVersions(GitAPI vc, ConfigTree ct) {
        this.vc = vc;
        this.ct = ct;
    }

    public boolean pull() {
        vc.pull();
        return true;
    }

    public GitAPI getGitAPI() {
        return vc;
    }

    @Override
    public String getEffectiveProperty(String hash, PosixPath leaf, String key) {
        return getHistorical(hash, () -> ct.getEffectiveProperty(leaf, key));
    }

    @Override
    public Set<Map.Entry<PosixPath, String>> getExplodedPropertyKeys(String hash) {
        return ct.getLeafs()
                .flatMap(this::flattenEffectiveProperties)
                .collect(Collectors.toSet());
    }

    private Stream<Map.Entry<PosixPath, String>> flattenEffectiveProperties(PosixPath leaf) {
        return getEffectivePropertyKeys(leaf)
                .stream()
                .map((k) -> new AbstractMap.SimpleEntry<>(leaf, k));
    }

    public Optional<String> setProperty(PosixPath leaf, String key, String value) {
        return ct.setProperty(leaf, key, value);
    }

    public Optional<String> deleteProperty(PosixPath leaf, String key) {
        return ct.deleteProperty(leaf, key);
    }

    public String getEffectiveProperty(PosixPath leaf, String key) {
        return ct.getEffectiveProperty(leaf, key);
    }

    public Set<String> getEffectivePropertyKeys(PosixPath leaf) {
        return ct.getEffectivePropertyKeys(leaf);
    }

    private <T> T getHistorical(String hash, Supplier<T> exec) {
        // String currHash = vc.listCommits().findFirst().get().getHash();
        String branch = vc.getBranch().get();

        try {
            vc.seek(hash);
            return exec.get();
        } finally {
            // return back on local HEAD
            // vc.seek(currHash);
            vc.reset();
            vc.setBranch(branch);
        }
    }

    @Override
    public Set<String> getEffectivePropertyKeys(String hash, PosixPath leaf) {
        return getHistorical(hash, () -> ct.getEffectivePropertyKeys(leaf));
    }

    public Stream<PosixPath> getLeafs() {
        return ct.getLeafs();
    }

    @Override
    public Stream<PropertiesVersion> listVersions() {
        return vc.listCommits()
                .map(c -> {
                    return PropertiesVersion.builder()
                            .versionHash(c.getHash())
                            .createdEpochSec(toEpochSeconds(c.getTimestamp()))
                            .build();
                });
    }

    public static long toEpochSeconds(String timestamp) {
        return Instant.from(DateTimeFormatter.ISO_DATE_TIME.parse(timestamp)).toEpochMilli() / 1000;
    }

    public static long getNowEpochSec() {
        return Instant.now().getEpochSecond();
    }


    public void commit() {
        vc.commit();
    }

    public void push() {
        vc.push();
    }

    @Override
    public void close() throws Exception {

    }

    @AllArgsConstructor
    @NoArgsConstructor
    @Builder
    @Data
    public static class BranchState {
        public String hash;
        public String status;
        public String executorId;
    }

}
