package io.github.andreyzebin.gitSql.git;

import io.github.zebin.javabash.process.TextTerminal;
import io.github.zebin.javabash.sandbox.BashUtils;
import io.github.zebin.javabash.sandbox.DirectoryTree;
import io.github.zebin.javabash.sandbox.FileManager;
import io.github.zebin.javabash.sandbox.WorkingDirectory;
import lombok.extern.slf4j.Slf4j;

import java.nio.file.Path;
import java.time.Instant;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Stream;

@Slf4j
public abstract class AbstractClient implements GitDirectory, GitAPI {

    protected final TextTerminal term;
    private Optional<String> origin = null;
    protected final StringBuffer stage = new StringBuffer();

    protected AbstractClient(TextTerminal term) {
        this.term = term;
    }

    /**
     * Fast-forward
     */
    @Override
    public Versions seek(String commit) {
        String pwd = term.eval("pwd");
        try {
            term.eval("cd " + BashUtils.encode(getRoot()));
            GitBindings.checkout(commit, term);
            return this;
        } finally {
            term.eval("cd " + pwd);
        }
    }

    @Override
    public void reset() {
        String pwd = term.eval("pwd");
        try {
            term.eval("cd " + BashUtils.encode(getRoot()));
            GitBindings.resetHard(term);
        } finally {
            term.eval("cd " + pwd);
        }
    }

    /**
     * Fast-forward
     */
    @Override
    public Stream<Commit> listCommits() {
        String pwd = term.eval("pwd");
        try {
            term.eval("cd " + BashUtils.encode(getRoot()));
            return GitBindings.commitsList(term);
        } finally {
            term.eval("cd " + pwd);
        }
    }

    @Override
    public Stream<BranchHead> listBranches() {
        String pwd = term.eval("pwd");
        try {
            term.eval("cd " + BashUtils.encode(getRoot()));
            return GitBindings.getBranches(term);
        } finally {
            term.eval("cd " + pwd);
        }
    }

    @Override
    public boolean contains(String hash) {
        Set<String> commits = new HashSet<>();

        listCommits().forEach(co -> {
            commits.add(co.getHash());
            commits.addAll(co.getParents());
        });
        return commits.contains(hash);
    }

    @Override
    public Stream<? extends Change> getChanges(String hashFrom) {
        String pwd = term.eval("pwd");
        try {
            term.eval("cd " + BashUtils.encode(getRoot()));
            return GitBindings.filesChangedQuery(
                    GitBindings.ALL_BRANCHES,
                    GitBindings.sinceHash(hashFrom),
                    term
            );
        } finally {
            term.eval("cd " + pwd);
        }
    }


    @Override
    public Stream<? extends Change> getChanges(String hashFrom, String hashTo) {
        String pwd = term.eval("pwd");
        try {
            term.eval("cd " + BashUtils.encode(getLocation().toPath()));
            return GitBindings.filesChangedQuery(
                    GitBindings.ALL_BRANCHES,
                    GitBindings.periodHash(hashFrom, hashTo),
                    term
            );
        } finally {
            term.eval("cd " + pwd);
        }
    }

    @Override
    public Instant getTimestamp() {
        String pwd = term.eval("pwd");
        try {
            term.eval("cd " + BashUtils.encode(getRoot()));
            return GitBindings.commitsList(term)
                    .findFirst()
                    .get()
                    .getTimestampInstant();
        } finally {
            term.eval("cd " + pwd);
        }
    }

    @Override
    public void pull() {
        String pwd = term.eval("pwd");
        try {
            term.eval("cd " + BashUtils.encode(getLocation().toPath()));
            GitBindings.pull(term);
        } finally {
            term.eval("cd " + pwd);
        }
    }

    @Override
    public void push() {
        String pwd = term.eval("pwd");
        try {
            term.eval("cd " + BashUtils.encode(getLocation().toPath()));
            GitBindings.push(term);
        } finally {
            term.eval("cd " + pwd);
        }
    }

    @Override
    public void setOrigin(String origin) {
        throw new RuntimeException("Unimplemented!");
    }

    @Override
    public Optional<String> getOrigin() {
        String pwd = term.eval("pwd");
        try {
            if (origin == null) {
                term.eval("cd " + BashUtils.encode(getRoot()));
                origin = GitBindings.getOrigin(term);
            }
            return origin;
        } finally {
            term.eval("cd " + pwd);
        }
    }

    @Override
    public void fetch() {
        String pwd = term.eval("pwd");
        try {
            term.eval("cd " + BashUtils.encode(getRoot()));
            term.eval(String.format("git fetch"));
        } finally {
            term.eval("cd " + pwd);
        }
    }

    @Override
    public void setUpstream(String localBranch, String upstream) {
        String pwd = term.eval("pwd");
        try {
            term.eval("cd " + BashUtils.encode(getRoot()));
            term.eval(String.format("git branch -u  %s %s ", upstream, localBranch));
        } finally {
            term.eval("cd " + pwd);
        }
    }

    @Override
    public Optional<String> getUpstream(String local) {
        throw new RuntimeException("Unimplemented!");
    }

    @Override
    public Optional<String> getBranch() {
        String pwd = term.eval("pwd");
        try {
            term.eval("cd " + BashUtils.encode(getRoot()));
            return GitBindings.getBranch(term);
        } finally {
            term.eval("cd " + pwd);
        }
    }

    /**
     * git checkout -b another_branch origin/another_branch; git branch -u origin/another_branch.
     * That's to create another_branch from origin/another_branch and set origin/another_branch
     * as the upstream of another_branch.
     *
     * @param branchName
     */
    @Override
    public void setBranch(String branchName) {
        String pwd = term.eval("pwd");
        try {
            term.eval("cd " + BashUtils.encode(getRoot()));
            term.eval(String.format("git checkout %s ", branchName));
        } finally {
            term.eval("cd " + pwd);
        }
    }

    @Override
    public void commit() {
        if (!stage.toString().isBlank()) {
            BashUtils.lockDir(
                    term,
                    () -> {
                        term.eval("cd " + BashUtils.encode(getRoot()));
                        Arrays.stream(stage.toString().split(System.lineSeparator()))
                                .forEach(cFile -> GitBindings.add(term, Path.of(cFile)));
                        if (GitBindings.hasStatus(term)) {
                            GitBindings.commit(term);
                        } else {
                            log.debug("Status has no changes, skip commit.");
                        }

                        return 0;
                    }
            );
        }

        stage.setLength(0);
    }

    public DirectoryTree getDirectory() {
        return new WorkingDirectory(
                new FileManager(term),
                getLocation(),
                (f) -> BashUtils.append(stage, f.getPath().toString())
        );
    }
}
