package io.github.andreyzebin.gitSql.cache;

import io.github.zebin.javabash.process.TextTerminal;
import io.github.zebin.javabash.sandbox.AllFileManager;
import io.github.zebin.javabash.sandbox.PosixPath;

import java.io.Writer;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;

public class FileManagerCacheProxy implements AllFileManager, GitEventListener {
    private final AtomicReference<String> cacheControl;
    private final Map<String, FileManagerCache> caches;
    private final Map<String, FileReadCache> readCaches;
    private final Map<String, LsCache> lsCache;
    private PosixPath cur;
    private final AllFileManager fm;

    public static FileManagerCacheProxy cachedProxy(
            AllFileManager delegate,
            AtomicReference<String> cacheControl
    ) {
        Map<String, FileManagerCache> fmCache = new HashMap<>();
        Map<String, FileReadCache> readCaches = new HashMap<>();
        Map<String, LsCache> lsCache = new HashMap<>();
        return new FileManagerCacheProxy(
                delegate,
                cacheControl,
                fmCache,
                readCaches,
                lsCache
        );
    }

    @Override
    public void call(String event) {
        if (event.equals("clear")) {
            readCaches.remove(cacheControl.get());
            caches.remove(cacheControl.get());
            lsCache.remove(cacheControl.get());
        }
    }

    @Override
    public TextTerminal getTerminal() {
        return fm.getTerminal();
    }

    public static class FileManagerCache {
        Map<PosixPath, Boolean> files = new HashMap<>();
        Map<PosixPath, Boolean> dirs = new HashMap<>();
    }

    public static class FileReadCache {
        Map<PosixPath, String> files = new HashMap<>();
    }

    public static class LsCache {
        Map<PosixPath, List<PosixPath>> files = new HashMap<>();
    }

    public FileManagerCacheProxy(
            AllFileManager delegate,
            AtomicReference<String> cacheControl,
            Map<String, FileManagerCache> caches,
            Map<String, FileReadCache> readCaches,
            Map<String, LsCache> lsCache
    ) {
        this.cacheControl = cacheControl;
        this.caches = caches;
        this.readCaches = readCaches;
        this.lsCache = lsCache;
        this.fm = delegate;
    }

    private void dropDir(PosixPath d) {
        getDirsCache().remove(d);
    }

    private void dropFile(PosixPath d) {
        getReadCache().remove(d);
        getFilesCache().remove(d);
    }

    private Map<PosixPath, Boolean> getDirsCache() {
        return getCache().dirs;
    }

    private Map<PosixPath, Boolean> getFilesCache() {
        return getCache().files;
    }

    private FileManagerCache getCache() {
        if (cacheControl.get() == null) {
            return new FileManagerCache();
        }

        return caches.computeIfAbsent(cacheControl.get(), h -> new FileManagerCache());
    }

    private Map<PosixPath, String> getReadCache() {
        if (cacheControl.get() == null) {
            return new FileReadCache().files;
        }

        return readCaches.computeIfAbsent(cacheControl.get(), h -> new FileReadCache()).files;
    }

    private Map<PosixPath, List<PosixPath>> getLsCache() {
        if (cacheControl.get() == null) {
            return new LsCache().files;
        }

        return lsCache.computeIfAbsent(cacheControl.get(), h -> new LsCache()).files;
    }

    @Override
    public String read(PosixPath pp) {
        return getReadCache().computeIfAbsent(getAbs(pp), p -> fm.read(pp));
    }

    @Override
    public List<PosixPath> list() {
        return List.of();
    }

    @Override
    public PosixPath getCurrent() {
        if (cur == null) {
            cur = fm.getCurrent();
            return cur;
        }
        return cur;
    }

    @Override
    public PosixPath goUp() {
        return fm.goUp();
    }

    @Override
    public PosixPath go(PosixPath path) {
        if (path.equals(cur)) {
            return path;
        }
        cur = fm.go(path);
        return cur;
    }

    @Override
    public PosixPath makeDir(PosixPath newDir) {
        if (dirExists(newDir)) {
            return getAbs(newDir);
        }
        getAbs(newDir).streamDescending().forEach(this::dropDir);
        return fm.makeDir(newDir);
    }

    @Override
    public PosixPath makeFile(PosixPath newDir) {
        dropFile(getAbs(newDir));

        return fm.makeFile(newDir);
    }

    @Override
    public Writer write(PosixPath pp) {
        dropFile(getAbs(pp));
        return fm.write(pp);
    }

    @Override
    public Writer append(PosixPath pp) {
        dropFile(getAbs(pp));
        return fm.append(pp);
    }

    @Override
    public boolean removeFile(PosixPath file) {
        dropFile(getAbs(file));
        return fm.removeFile(file);
    }

    @Override
    public boolean removeDir(PosixPath path) {
        getFilesCache()
                .keySet()
                .stream()
                .filter(kk -> kk.startsWith(getAbs(path)))
                .forEach(kk -> getFilesCache().remove(kk));

        getReadCache()
                .keySet()
                .stream()
                .filter(kk -> kk.startsWith(getAbs(path)))
                .forEach(kk -> getReadCache().remove(kk));

        getDirsCache()
                .keySet()
                .stream()
                .filter(kk -> kk.startsWith(getAbs(path)))
                .forEach(kk -> getDirsCache().remove(kk));
        return fm.removeDir(path);
    }

    @Override
    public boolean dirExists(PosixPath newDir) {
        return getDirsCache().computeIfAbsent(getAbs(newDir),
                d -> fm.dirExists(newDir));
    }

    private PosixPath getAbs(PosixPath newDir) {
        return newDir.isAbsolute() ? newDir : getCurrent().climb(newDir);
    }

    @Override
    public boolean fileExists(PosixPath newDir) {
        return getFilesCache().computeIfAbsent(getAbs(newDir), f -> fm.fileExists(newDir));
    }

    @Override
    public List<PosixPath> list(PosixPath path) {
        return getLsCache().computeIfAbsent(getAbs(path), p -> fm.list(path));
    }
}
