package io.github.andreyzebin.gitSql.git;

import io.github.andreyzebin.gitSql.config.DirectoryTreeFactory;
import io.github.zebin.javabash.sandbox.*;
import lombok.extern.slf4j.Slf4j;

import java.nio.file.Path;
import java.time.Instant;
import java.util.*;
import java.util.function.Supplier;
import java.util.stream.Stream;

@Slf4j
public abstract class AbstractClient implements GitFs {

    protected final AllFileManager fm;
    protected final StringBuffer stage = new StringBuffer();
    protected final DirectoryTreeFactory<DirectoryTree> dtFactory;
    protected final Map<String, DirectoryTree> dt = new HashMap<>();

    protected AbstractClient(AllFileManager fm, DirectoryTreeFactory<DirectoryTree> dtFactory) {
        this.fm = fm;
        this.dtFactory = dtFactory;
    }

    /**
     * Fast-forward
     */
    @Override
    public Versions seek(String commit) {
        return setupDir(() -> {
            GitBindings.checkout(commit, fm.getTerminal());
            return this;
        });
    }

    @Override
    public void reset() {
        setupDir(() -> GitBindings.resetHard(fm.getTerminal()));
    }

    /**
     * Fast-forward
     */
    @Override
    public Stream<Commit> listCommits() {
        return setupDir(() -> GitBindings.commitsList(fm.getTerminal()));
    }

    @Override
    public Stream<BranchHead> listBranches() {
        return setupDir(() -> GitBindings.getBranches(fm.getTerminal()));
    }

    @Override
    public boolean contains(String hash) {
        Set<String> commits = new HashSet<>();

        listCommits().forEach(co -> {
            commits.add(co.getHash());
            commits.addAll(co.getParents());
        });
        return commits.contains(hash);
    }

    @Override
    public Stream<? extends Change> getChanges(String hashFrom) {
        return setupDir(() -> GitBindings.filesChangedQuery(
                GitBindings.ALL_BRANCHES,
                GitBindings.sinceHash(hashFrom),
                fm.getTerminal()
        ));
    }


    @Override
    public Stream<? extends Change> getChanges(String hashFrom, String hashTo) {
        return setupDir(() -> GitBindings.filesChangedQuery(
                GitBindings.ALL_BRANCHES,
                GitBindings.periodHash(hashFrom, hashTo),
                fm.getTerminal()
        ));
    }

    @Override
    public Instant getTimestamp() {
        return setupDir(() -> GitBindings.commitsList(fm.getTerminal())
                .findFirst()
                .get()
                .getTimestampInstant());
    }

    @Override
    public void pull() {
        setupDir(() -> GitBindings.pull(fm.getTerminal()));
    }

    @Override
    public void push() {
        setupDir(() -> GitBindings.push(fm.getTerminal()));
    }

    @Override
    public void setOrigin(String origin) {
        throw new RuntimeException("Unimplemented!");
    }

    @Override
    public Optional<String> getOrigin() {
        return setupDir(() -> GitBindings.getOrigin(fm.getTerminal()));
    }

    @Override
    public void fetch() {
        setupDir(() -> {
            fm.getTerminal().eval(String.format("git fetch"));
        });
    }

    @Override
    public void setUpstream(String localBranch, String upstream) {
        setupDir(() -> {
            fm.getTerminal().eval(String.format("git branch -u  %s %s ", upstream, localBranch));
        });
    }

    @Override
    public Optional<String> getUpstream(String local) {
        throw new RuntimeException("Unimplemented!");
    }

    @Override
    public Optional<String> getBranch() {
        return setupDir(() -> GitBindings.getBranch(fm.getTerminal()));
    }

    /**
     * git checkout -b another_branch origin/another_branch; git branch -u origin/another_branch.
     * That's to create another_branch from origin/another_branch and set origin/another_branch
     * as the upstream of another_branch.
     * <p>
     * TODO
     *
     * @param branchName
     */
    @Override
    public void setBranch(String branchName) {
        setupDir(() -> {
            fm.getTerminal().eval(String.format("git checkout %s ", branchName));
        });
    }

    @Override
    public void merge(String hash) {
        setupDir(() -> {
            fm.getTerminal().eval(String.format("git merge %s ", hash));
        });
    }

    @Override
    public void commit() {
        if (!stage.toString().isBlank()) {
            setupDir(() -> {
                        Arrays.stream(stage.toString().split(System.lineSeparator()))
                                .forEach(cFile -> GitBindings.add(fm.getTerminal(), Path.of(cFile)));
                        if (GitBindings.hasStatus(fm.getTerminal())) {
                            GitBindings.commit(fm.getTerminal());
                        } else {
                            log.debug("Status has no changes, skip commit.");
                        }
                    }
            );
        }

        stage.setLength(0);
    }

    @Override
    public DirectoryTree getDirectory() {
        return dt.computeIfAbsent("dt", d -> {
            return dtFactory.produce(
                    fm,
                    getLocation(),
                    (f) -> BashUtils.append(stage, f.getPath().toString())
            );
        });
    }

    protected <T> T recoverState(Supplier<T> result) {
        String corrId = UUID.randomUUID().toString().substring(1, 3);
        log.debug("File manager state #{} saving...", corrId);
        PosixPath pwd = fm.getCurrent();
        try {
            return result.get();
        } finally {
            fm.go(pwd);
            log.debug("File manager state #{} recovered.", corrId);
        }
    }

    protected <T> T setupDir(Supplier<T> result) {
        String corrId = UUID.randomUUID().toString().substring(1, 3);
        log.debug("File manager state #{} saving...", corrId);
        PosixPath pwd = fm.getCurrent();
        try {
            if (!pwd.equals(getLocation())) {
                fm.makeDir(getLocation());
                fm.go(getLocation());
            }
            return result.get();
        } finally {
            if (!pwd.equals(getLocation())) {
                fm.go(pwd);
            }
            log.debug("File manager state #{} recovered.", corrId);
        }
    }

    protected void setupDir(Runnable result) {
        setupDir(() -> {
            result.run();
            return 0;
        });
    }
}
