package io.github.andreyzebin.gitSql.cache;

import io.github.andreyzebin.gitSql.FileSystemUtils;
import io.github.zebin.javabash.sandbox.DirectoryTree;
import io.github.zebin.javabash.sandbox.PosixPath;
import lombok.extern.slf4j.Slf4j;

import java.io.Reader;
import java.io.StringReader;
import java.io.Writer;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Stream;


@Slf4j
public class DirectoryTreeCacheProxy implements DirectoryTree, GitEventListener {
    private final DirectoryTree dt;
    private final AtomicReference<String> cacheControl;
    private final Map<String, FileManagerCacheProxy.FileManagerCache> caches;
    private final Map<String, FileManagerCacheProxy.FileReadCache> readCaches;
    private final Map<String, FileManagerCacheProxy.LsCache> lsCache;

    public DirectoryTreeCacheProxy(
            DirectoryTree dt,
            AtomicReference<String> cacheControl,
            Map<String, FileManagerCacheProxy.FileManagerCache> caches,
            Map<String, FileManagerCacheProxy.FileReadCache> readCaches,
            Map<String, FileManagerCacheProxy.LsCache> lsCache
    ) {
        this.dt = dt;
        this.cacheControl = cacheControl;
        this.caches = caches;
        this.readCaches = readCaches;
        this.lsCache = lsCache;
    }

    public static DirectoryTreeCacheProxy cachedProxy(
            DirectoryTree delegate,
            AtomicReference<String> cacheControl
    ) {
        Map<String, FileManagerCacheProxy.FileManagerCache> fmCache = new HashMap<>();
        Map<String, FileManagerCacheProxy.FileReadCache> readCaches = new HashMap<>();
        Map<String, FileManagerCacheProxy.LsCache> lsCache = new HashMap<>();
        return new DirectoryTreeCacheProxy(
                delegate,
                cacheControl,
                fmCache,
                readCaches,
                lsCache
        );
    }

    @Override
    public void call(String event) {
        if (event.equals("clear")) {
            readCaches.remove(cacheControl.get());
            caches.remove(cacheControl.get());
            lsCache.remove(cacheControl.get());
        }
    }

    private void dropDir(PosixPath d) {
        getDirsCache().remove(d);
    }

    private void dropFile(PosixPath d) {
        getReadCache().remove(d);
        getFilesCache().remove(d);
    }

    private Map<PosixPath, Boolean> getDirsCache() {
        return getCache().dirs;
    }

    private Map<PosixPath, Boolean> getFilesCache() {
        return getCache().files;
    }

    private FileManagerCacheProxy.FileManagerCache getCache() {
        if (cacheControl.get() == null) {
            return new FileManagerCacheProxy.FileManagerCache();
        }

        return caches.computeIfAbsent(cacheControl.get(), h -> new FileManagerCacheProxy.FileManagerCache());
    }

    private Map<PosixPath, String> getReadCache() {
        if (cacheControl.get() == null) {
            return new FileManagerCacheProxy.FileReadCache().files;
        }

        return readCaches.computeIfAbsent(cacheControl.get(), h -> new FileManagerCacheProxy.FileReadCache()).files;
    }

    private Map<PosixPath, List<PosixPath>> getLsCache() {
        if (cacheControl.get() == null) {
            return new FileManagerCacheProxy.LsCache().files;
        }

        return lsCache.computeIfAbsent(cacheControl.get(), h -> new FileManagerCacheProxy.LsCache()).files;
    }

    @Override
    public Writer put(PosixPath posixPath) {
        posixPath.streamDescending().forEach(this::dropDir);
        dropFile(posixPath);

        return dt.put(posixPath);
    }

    @Override
    public boolean delete(PosixPath posixPath) {
        getFilesCache()
                .keySet()
                .stream()
                .filter(kk -> kk.startsWith(posixPath))
                .forEach(kk -> getFilesCache().remove(kk));

        getReadCache()
                .keySet()
                .stream()
                .filter(kk -> kk.startsWith(posixPath))
                .forEach(kk -> getReadCache().remove(kk));

        getDirsCache()
                .keySet()
                .stream()
                .filter(kk -> kk.startsWith(posixPath))
                .forEach(kk -> getDirsCache().remove(kk));

        return false;
    }

    @Override
    public Writer patch(PosixPath posixPath) {
        posixPath.streamDescending().forEach(this::dropDir);
        dropFile(posixPath);

        return dt.patch(posixPath);
    }

    @Override
    public Reader get(PosixPath posixPath) {
        return new StringReader(getReadCache()
                .computeIfAbsent(posixPath, p -> FileSystemUtils.loadFile(dt.get(posixPath))));
    }

    @Override
    public boolean exists(PosixPath posixPath) {
        Boolean dirExists = getDirsCache()
                .computeIfAbsent(posixPath, d -> dt.isDir(posixPath));

        Boolean fileExists = getFilesCache()
                .computeIfAbsent(posixPath, f -> dt.exists(posixPath) && !dirExists);

        return fileExists || dirExists;
    }

    @Override
    public boolean isDir(PosixPath posixPath) {
        return getDirsCache().computeIfAbsent(posixPath, d -> dt.isDir(posixPath));
    }

    @Override
    public Stream<PosixPath> list(PosixPath posixPath) {
        return getLsCache()
                .computeIfAbsent(posixPath, p -> dt.list(posixPath).toList())
                .stream();
    }

    @Override
    public void traverse(PosixPath start, Function<PosixPath, Boolean> sayEnter) {
        dt.traverse(start, sayEnter);
    }

    @Override
    public <T> T setupDir(Supplier<T> result) {
        return dt.setupDir(result);
    }
}
