package io.github.andreyzebin.gitSql.sql;

import io.github.andreyzebin.gitSql.FileSystem;
import java.io.Reader;
import java.nio.file.Path;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Map;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.stream.Collectors;

import io.github.andreyzebin.gitSql.FileSystemUtils;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public class FilesIndex extends PersistedIndex {

    private final FileSystem fileSystem;
    private final Map<Function<String, Boolean>, BiFunction<String, Reader, String>> mappers;

    public static final Map<Function<String, Boolean>, BiFunction<String, Reader, String>> JSON_MAPPERS =
            Map.of(
                    (f) -> f.endsWith(".json"),
                    (f, r) -> FileSystemUtils.readLines(r).collect(Collectors.joining(System.lineSeparator()))
            );

    public FilesIndex(
            FileSystem fileSystem,
            Path storeDir,
            Map<Function<String, Boolean>, BiFunction<String, Reader, String>> mappers
    ) {
        super(storeDir);
        this.fileSystem = fileSystem;
        this.mappers = mappers;
    }

    public static FilesIndex of(
            FileSystem fileSystem,
            Path storeDir,
            Map<Function<String, Boolean>, BiFunction<String, Reader, String>> mappers) {
        return new FilesIndex(fileSystem, storeDir, mappers);
    }

    public static BiFunction<FileSystem, Path, FilesIndex> of(
            Map<Function<String, Boolean>, BiFunction<String, Reader, String>> mappers
    ) {
        return (a, b) -> new FilesIndex(a, b, mappers);
    }

    @Override
    protected void createSchema() throws SQLException {
        try (Statement dml = getConnection().createStatement();) {
            final String sql = """
                    CREATE TABLE IF NOT EXISTS files
                    (
                        ID      INT          auto_increment,
                        path    VARCHAR(256) NOT NULL,
                        data    JSON         NOT NULL,
                        UNIQUE NULLS ALL DISTINCT (path)
                    );""";

            log.debug(CommitsIndex.renderSqlLog(sql));
            dml.execute(sql);
        }
        super.createSchema();
    }

    @Override
    protected void reindex() {
        try {
            fileSystem.find(
                    (p) -> {
                        if (!fileSystem.isDir(p)) {
                            final String fileName = p.getFileName().toString();
                            mappers.entrySet()
                                    .stream()
                                    .filter(cKey -> cKey.getKey().apply(fileName))
                                    .findFirst()
                                    .ifPresent(
                                            cF -> SqlUtils.merge(
                                                    getConnection(),
                                                    "files",
                                                    "path",
                                                    "data FORMAT JSON",
                                                    p.toString(),
                                                    cF.getValue().apply(fileName, fileSystem.get(p))
                                            )
                                    );
                        }
                        return true;
                    }
            );
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

}
