package io.github.andreyzebin.gitSql.sql;

import io.github.andreyzebin.gitSql.git.VersionedFiles;
import io.github.andreyzebin.gitSql.sql.SqlUtils.SchemaBuilder;
import java.sql.SQLException;
import java.time.Instant;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.function.Supplier;
import java.util.stream.Stream;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public class RangeIndexMerger implements TableSource {

    private final Supplier<TableSource> delegate;
    private final Instant from;
    private final Instant to;
    private final VersionedFiles dataSet;
    private final TableSource cachedDelegate;

    public RangeIndexMerger(
            Supplier<TableSource> delegate,
            Instant from,
            Instant instant,
            VersionedFiles dataSet
    ) {
        this.delegate = delegate;
        this.from = from;
        this.to = instant;
        this.dataSet = dataSet;
        this.cachedDelegate = delegate.get();
    }

    @Override
    public SchemaBuilder createSchema() throws SQLException {
        return delegate.get()
                .createSchema()
                .withColumn(TimeSeriesQuery.COLUMN_NAME_POINT)
                .withColumn(TimeSeriesQuery.COLUMN_NAME_HASH)
                .withColumn(TimeSeriesQuery.COLUMN_NAME_ORIGIN)
                .withMerge(TimeSeriesQuery.COLUMN_NAME_POINT);
    }

    @Override
    public Stream<Map<String, String>> rows() {
        return GitUtils.getPoints(dataSet, from, to)
                .flatMap(
                        commit -> {
                            dataSet.getSource().seek(commit.getHash());

                            return delegate.get()
                                    .rows()
                                    .map(cRow -> {
                                        cRow = new HashMap<>(cRow);
                                        cRow.put(TimeSeriesQuery.COLUMN_NAME_POINT, commit.getTimestamp());
                                        cRow.put(TimeSeriesQuery.COLUMN_NAME_HASH, commit.getHash());
                                        cRow.put(TimeSeriesQuery.COLUMN_NAME_ORIGIN,
                                                dataSet.getSource().getOrigin().orElse(dataSet.getRoot().toString()));
                                        return cRow;
                                    });

                        }
                );
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        RangeIndexMerger that = (RangeIndexMerger) o;
        return Objects.equals(from, that.from) && Objects.equals(to, that.to)
                && Objects.equals(cachedDelegate, that.cachedDelegate);
    }

    @Override
    public int hashCode() {
        return Objects.hash(from, to, cachedDelegate, this.getClass().getName());
    }
}
