package io.github.devopsplugin.git.helper;

import io.github.devopsplugin.git.vo.BlobWrapper;
import io.github.devopsplugin.git.vo.DiffEntryWrapper;
import org.eclipse.jgit.api.Git;
import org.eclipse.jgit.api.Status;
import org.eclipse.jgit.diff.*;
import org.eclipse.jgit.dircache.DirCache;
import org.eclipse.jgit.dircache.DirCacheIterator;
import org.eclipse.jgit.errors.AmbiguousObjectException;
import org.eclipse.jgit.errors.LargeObjectException;
import org.eclipse.jgit.errors.MissingObjectException;
import org.eclipse.jgit.lib.*;
import org.eclipse.jgit.revwalk.RevCommit;
import org.eclipse.jgit.revwalk.RevWalk;
import org.eclipse.jgit.treewalk.AbstractTreeIterator;
import org.eclipse.jgit.treewalk.CanonicalTreeParser;
import org.eclipse.jgit.treewalk.TreeWalk;
import org.eclipse.jgit.treewalk.filter.OrTreeFilter;
import org.eclipse.jgit.treewalk.filter.PathFilter;
import org.eclipse.jgit.treewalk.filter.TreeFilter;

import java.io.File;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.*;
import java.util.stream.Collectors;

public class DiffHelper {

    private static final DiffAlgorithm DIFF_ALGORITHM = new HistogramDiff();
    private static final Field DIFF_ENTRY_OLD_ID_FIELD = getField(DiffEntry.class, "oldId");
    private static final Field DIFF_ENTRY_NEW_ID_FIELD = getField(DiffEntry.class, "newId");
    private static final Method DIFF_ENTRY_ADD_METHOD = getDiffEntryAddMethod();
    private static final Method DIFF_ENTRY_MODIFY_METHOD = getDiffEntryModifyMethod();
    private static final byte[] EMPTY = new byte[]{};
    private static final byte[] BINARY = new byte[]{};
    private static final RawTextComparator RAW_TEXT_COMPARATOR = RawTextComparator.DEFAULT;
    private static final int BIG_FILE_THRESHOLD = 10 * 1024 * 1024;

    public static List<DiffEntryWrapper> calculateDiff(File gitDir, String oldRev, String newRev, boolean includeStagedCodes) throws Exception {
        try (Git git = Git.open(gitDir)) {
            ObjectReader reader = git.getRepository().newObjectReader();
            RevWalk revWalk = new RevWalk(git.getRepository());

            RevCommit oldCommit = revWalk.parseCommit(git.getRepository().resolve(oldRev));
            RevCommit newCommit = revWalk.parseCommit(git.getRepository().resolve(newRev));

            List<DiffEntryWrapper> wrappers = new ArrayList<>();
            if (includeStagedCodes) {
                wrappers.addAll(doCalculateIndexedDiff(git, oldCommit, reader, gitDir));
            }

            Set<String> indexedPaths = wrappers.stream()
                    .map(DiffEntryWrapper::getNewPath)
                    .collect(Collectors.toSet());

            wrappers.addAll(doCalculateCommitDiff(oldCommit, newCommit, reader, git, gitDir, indexedPaths));
            return wrappers;
        }
    }

    private static List<DiffEntryWrapper> doCalculateCommitDiff(RevCommit oldCommit, RevCommit newCommit,
                                                                ObjectReader reader, Git git, File gitDir,
                                                                Collection<String> excludedPaths) throws Exception {
        if (Objects.equals(oldCommit.getId(), newCommit.getId())) {
            return Collections.emptyList();
        }

        if (Objects.equals(oldCommit.getTree().getId(), newCommit.getTree().getId())) {
            return Collections.emptyList();
        }

        RenameDetector detector = new RenameDetector(git.getRepository());
        AbstractTreeIterator oldTree = new CanonicalTreeParser(null, reader, oldCommit.getTree());
        AbstractTreeIterator newTree = new CanonicalTreeParser(null, reader, newCommit.getTree());

        List<DiffEntry> entries = git.diff().setOldTree(oldTree).setNewTree(newTree).call();
        detector.reset();
        detector.addAll(entries);
        entries = detector.compute();

        return entries.stream()
                .filter(entry -> !excludedPaths.contains(entry.getNewPath()))
                .map(entry -> {
                    RawText oldText = newRawText(entry, DiffEntry.Side.OLD, reader);
                    RawText newText = newRawText(entry, DiffEntry.Side.NEW, reader);
                    return DiffEntryWrapper.builder()
                            .gitDir(gitDir)
                            .diffEntry(entry)
                            .edits(calculateEditList(oldText, newText))
                            .build();
                }).collect(Collectors.toList());
    }

    private static List<DiffEntryWrapper> doCalculateIndexedDiff(Git git, RevCommit oldCommit, ObjectReader reader, File gitDir) throws Exception {
        Set<String> indexedPathSet = new HashSet<>();
        Status status = git.status().call();
        indexedPathSet.addAll(status.getAdded());
        indexedPathSet.addAll(status.getChanged());
        Map<String, BlobWrapper> indexedFileContentMap = getIndexedFileContentMap(git, indexedPathSet);
        Map<String, BlobWrapper> oldRevFileContentMap = getRevFileContentMap(git, oldCommit, indexedPathSet, reader);
        return indexedPathSet.stream()
                .map(filePath -> {
                    BlobWrapper oldBlob = oldRevFileContentMap.get(filePath);
                    RawText oldText = oldBlob != null ? new RawText(oldBlob.getContent()) : RawText.EMPTY_TEXT;
                    RawText newText = new RawText(indexedFileContentMap.get(filePath).getContent());
                    DiffEntry entry = oldBlob == null ? createAddDiffEntry(filePath, oldCommit) : createModifyDiffEntry(filePath);
                    return DiffEntryWrapper.builder()
                            .gitDir(gitDir)
                            .diffEntry(entry)
                            .edits(calculateEditList(oldText, newText))
                            .build();
                })
                .collect(Collectors.toList());
    }

    private static DiffEntry createAddDiffEntry(String path, AnyObjectId id) {
        try {
            return (DiffEntry) DIFF_ENTRY_ADD_METHOD.invoke(null, path, id);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private static DiffEntry createModifyDiffEntry(String path) {
        try {
            return (DiffEntry) DIFF_ENTRY_MODIFY_METHOD.invoke(null, path);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private static Map<String, BlobWrapper> getRevFileContentMap(Git git, RevCommit commit, Set<String> filePaths, ObjectReader reader) throws Exception {
        if (filePaths == null || filePaths.isEmpty()) {
            return Collections.emptyMap();
        }
        TreeFilter filter = filePaths.size() > 1
                ? OrTreeFilter.create(filePaths.stream()
                .map(PathFilter::create)
                .collect(Collectors.toList()))
                : PathFilter.create(filePaths.iterator().next());
        return getContentMapByTreeAndFilter(git, new CanonicalTreeParser(null, reader, commit.getTree()), filter);
    }

    private static Map<String, BlobWrapper> getIndexedFileContentMap(Git git, Set<String> filePaths) throws Exception {
        if (filePaths == null || filePaths.isEmpty()) {
            return Collections.emptyMap();
        }
        DirCache index = git.getRepository().readDirCache();
        TreeFilter filter = filePaths.size() > 1
                ? OrTreeFilter.create(filePaths.stream()
                .map(PathFilter::create)
                .collect(Collectors.toList()))
                : PathFilter.create(filePaths.iterator().next());
        return getContentMapByTreeAndFilter(git, new DirCacheIterator(index), filter);
    }

    private static Map<String, BlobWrapper> getContentMapByTreeAndFilter(Git git, AbstractTreeIterator tree, TreeFilter filter) throws Exception {
        Map<String, BlobWrapper> contentMap = new LinkedHashMap<>();
        try (TreeWalk treeWalk = new TreeWalk(git.getRepository())) {
            treeWalk.addTree(tree);
            treeWalk.setRecursive(true);
            treeWalk.setFilter(filter);
            while (treeWalk.next()) {
                ObjectId objectId = treeWalk.getObjectId(0);
                ObjectLoader loader = git.getRepository().open(objectId);
                BlobWrapper blobWrapper = BlobWrapper.builder()
                        .blobId(objectId)
                        .content(loader.getBytes())
                        .build();
                contentMap.put(treeWalk.getPathString(), blobWrapper);
            }
        }
        return contentMap;
    }

    private static RawText newRawText(DiffEntry entry, DiffEntry.Side side, ObjectReader reader) {
        try {
            return new RawText(open(entry, side, reader, BIG_FILE_THRESHOLD));
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private static List<Edit> calculateEditList(RawText oldText, RawText newText) {
        return DIFF_ALGORITHM.diff(RAW_TEXT_COMPARATOR, oldText, newText);
    }

    private static byte[] open(DiffEntry entry, DiffEntry.Side side, ObjectReader reader, int bigFileThreshold) throws Exception {
        if (entry.getMode(side) == FileMode.GITLINK) {
            return writeGitLinkText(entry.getId(side));
        }
        if (entry.getMode(side) == FileMode.MISSING) {
            return EMPTY;
        }
        if (entry.getMode(side).getObjectType() != Constants.OBJ_BLOB) {
            return EMPTY;
        }
        AbbreviatedObjectId id = entry.getId(side);
        if (!id.isComplete()) {
            Collection<ObjectId> ids = reader.resolve(id);
            if (ids.size() == 1) {
                id = AbbreviatedObjectId.fromObjectId(ids.iterator().next());
                switch (side) {
                    case OLD:
                        DIFF_ENTRY_OLD_ID_FIELD.set(entry, id);
                        break;
                    case NEW:
                        DIFF_ENTRY_NEW_ID_FIELD.set(entry, id);
                        break;
                    default:
                        break;
                }
            } else if (ids.size() == 0) {
                throw new MissingObjectException(id, Constants.OBJ_BLOB);
            } else {
                throw new AmbiguousObjectException(id, ids);
            }
        }
        ContentSource cs = ContentSource.create(reader);
        try {
            ObjectLoader ldr = new ContentSource.Pair(cs, cs).open(side, entry);
            return ldr.getBytes(bigFileThreshold);

        } catch (LargeObjectException.ExceedsLimit overLimit) {
            return BINARY;

        } catch (LargeObjectException.ExceedsByteArrayLimit overLimit) {
            return BINARY;

        } catch (LargeObjectException.OutOfMemory tooBig) {
            return BINARY;

        } catch (LargeObjectException tooBig) {
            tooBig.setObjectId(id.toObjectId());
            throw tooBig;
        }
    }

    private static byte[] writeGitLinkText(AbbreviatedObjectId id) {
        if (ObjectId.zeroId().equals(id.toObjectId())) {
            return EMPTY;
        }
        return Constants.encodeASCII("Subproject commit " + id.name() + "\n");
    }

    private static <T> Field getField(Class<T> clazz, String fieldName) {
        try {
            Field field = clazz.getDeclaredField(fieldName);
            field.setAccessible(true);
            return field;
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private static Method getDiffEntryAddMethod() {
        Method method;
        try {
            method = DiffEntry.class.getDeclaredMethod("add", String.class, AnyObjectId.class);
            method.setAccessible(true);
            return method;
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private static Method getDiffEntryModifyMethod() {
        Method method;
        try {
            method = DiffEntry.class.getDeclaredMethod("modify", String.class);
            method.setAccessible(true);
            return method;
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

}
