package io.github.devopsplugin.jacoco.filter;

import io.github.devopsplugin.jacoco.util.CollectionUtil;
import io.github.devopsplugin.jacoco.util.FilterUtil;
import io.github.devopsplugin.jacoco.util.LineNumberNodeWrapper;
import org.apache.maven.project.MavenProject;
import org.codehaus.plexus.util.StringUtils;
import org.eclipse.jgit.blame.BlameResult;
import org.eclipse.jgit.lib.PersonIdent;
import org.jacoco.core.internal.analysis.filter.IFilter;
import org.jacoco.core.internal.analysis.filter.IFilterContext;
import org.jacoco.core.internal.analysis.filter.IFilterOutput;
import org.objectweb.asm.tree.InsnList;
import org.objectweb.asm.tree.MethodNode;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class PersonFilter implements IFilter {

    private static final String SOURCE_PATH_PREFIX = "/src/main/java/";
    private final Map<String, BlameResult> classPathBlameResultMap = new HashMap<>();
    private final PersonInfo personInfo;

    public PersonFilter(MavenProject project, PersonInfo personInfo, List<BlameResult> blameResults) {
        List<String> modules = project.getModules();
        if (project.getParent() != null && CollectionUtil.isNotEmpty(project.getParent().getModules())) {
            modules.addAll(project.getParent().getModules());
        }

        for (BlameResult blameResult : blameResults) {
            String name = blameResult.getResultPath();
            if (CollectionUtil.isNotEmpty(modules)) {
                for (String module : modules) {
                    if (name.startsWith(module)) {
                        name = StringUtils.replaceOnce(name, module, "");
                        break;
                    }
                }
            }

            if (!name.startsWith(SOURCE_PATH_PREFIX)) {
                continue;
            }
            name = StringUtils.replaceOnce(name, SOURCE_PATH_PREFIX, "");
            classPathBlameResultMap.put(name, blameResult);
        }

        this.personInfo = personInfo;
    }

    @Override
    public void filter(MethodNode methodNode, IFilterContext context, IFilterOutput output) {
        if (personInfo == null) {
            return;
        }

        String classPath = FilterUtil.getClassPath(context);
        InsnList instructions = methodNode.instructions;
        BlameResult blameResult = classPathBlameResultMap.get(classPath);
        if (blameResult == null) {
            output.ignore(instructions.getFirst(), instructions.getLast());
            return;
        }

        List<LineNumberNodeWrapper> nodeWrappers = FilterUtil.collectLineNumberNodeList(instructions);
        for (LineNumberNodeWrapper nodeWrapper : nodeWrappers) {
            int line = nodeWrapper.getLine() - 1;
            PersonIdent personIdent = personInfo.getType().getPerson(blameResult, line);
            if (personInfo.accept(personIdent)) {
                nodeWrapper.setIgnored(false);
            }
        }

        for (LineNumberNodeWrapper nodeWrapper : nodeWrappers) {
            if (nodeWrapper.isIgnored()) {
                output.ignore(nodeWrapper.getNode(), nodeWrapper.getNext());
            }
        }
    }

    public static class PersonInfo {

        private String name;
        private String email;
        private PersonType type;

        public PersonInfo(String name, String email, PersonType type) {
            this.name = name;
            this.email = email;
            this.type = type;
        }

        public String getName() {
            return name;
        }

        public String getEmail() {
            return email;
        }

        public PersonType getType() {
            return type;
        }

        public boolean accept(PersonIdent personIdent) {
            if (personIdent == null) {
                return false;
            }

            if (StringUtils.isNotBlank(name) && !StringUtils.equals(name, personIdent.getName())) {
                return false;
            }

            if (StringUtils.isNotBlank(email) && !StringUtils.equals(email, personIdent.getEmailAddress())) {
                return false;
            }

            return true;
        }

    }

    public static enum PersonType {

        AUTHOR {
            @Override
            public PersonIdent getPerson(BlameResult blameResult, int line) {
                return blameResult.getSourceAuthor(line);
            }
        };

        public abstract PersonIdent getPerson(BlameResult blameResult, int line);

    }
}
