package io.github.devopsplugin.jacoco.maven;

import io.github.devopsplugin.git.helper.BlameHelper;
import io.github.devopsplugin.git.helper.DiffHelper;
import io.github.devopsplugin.git.vo.DiffEntryWrapper;
import io.github.devopsplugin.jacoco.filter.DiffFilter;
import io.github.devopsplugin.jacoco.filter.PersonFilter;
import io.github.devopsplugin.jacoco.util.CollectionUtil;
import io.github.devopsplugin.jacoco.util.FilterUtil;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.plugins.annotations.ResolutionScope;
import org.codehaus.plexus.util.StringUtils;
import org.eclipse.jgit.blame.BlameResult;
import org.eclipse.jgit.storage.file.FileRepositoryBuilder;
import org.jacoco.core.internal.analysis.filter.IFilter;
import org.jacoco.maven.AgentMojo;

import java.io.File;
import java.util.List;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;

@Mojo(
        name = "prepare-agent",
        defaultPhase = LifecyclePhase.INITIALIZE,
        requiresDependencyResolution = ResolutionScope.RUNTIME,
        threadSafe = true
)
public class DiffAgentMojo extends AgentMojo {

    private static final AtomicBoolean LOCK = new AtomicBoolean(false);
    private static final String GIT_REGEX = "\\.git$";

    @Parameter(property = "jacoco.git.oldrev", defaultValue = "")
    private String oldRev;
    @Parameter(property = "jacoco.git.newrev", defaultValue = "")
    private String newRev;

    @Parameter(property = "jacoco.author.name", defaultValue = "")
    private String authorName;
    @Parameter(property = "jacoco.author.email", defaultValue = "")
    private String authorEmail;

    @Override
    public void executeMojo() {
        try {
            if ((StringUtils.isNotBlank(oldRev) && StringUtils.isNotBlank(newRev))) {
                getLog().info("start execute diff filter for old rev [" + oldRev + "] and new rev [" + newRev + "]");
                doDiffFilter();
                getLog().info("finished execute diff filter for old rev [" + oldRev + "] and new rev [" + newRev + "]");
            }
        } catch (Exception e) {
            getLog().error("failed to execute diff filter for old rev [" + oldRev + "] and new rev [" + newRev + "]");
        }
        super.executeMojo();
    }

    private void doDiffFilter() throws Exception {
        if (LOCK.getAndSet(true)) {
            return;
        }

        File baseDir = getProject().getBasedir();
        File gitDir = new FileRepositoryBuilder().findGitDir(baseDir).getGitDir();
        gitDir = new File(gitDir.getAbsolutePath().replaceAll(GIT_REGEX, ""));

        List<DiffEntryWrapper> diffEntries = DiffHelper.calculateDiff(gitDir, oldRev, newRev, false)
                .stream()
                .filter(diffEntry -> !diffEntry.isDeleted())
                .collect(Collectors.toList());
        if (CollectionUtil.isEmpty(diffEntries)) {
            return;
        }

        IFilter diffFilter = new DiffFilter(getProject(), diffEntries);
        FilterUtil.appendFilter(diffFilter);

        if (StringUtils.isNotBlank(authorName) || StringUtils.isNotBlank(authorEmail)) {
            Set<String> filePaths = diffEntries.stream().map(DiffEntryWrapper::getNewPath).collect(Collectors.toSet());
            List<BlameResult> blameResults = BlameHelper.getBlames(gitDir, filePaths, newRev);
            PersonFilter.PersonInfo author = new PersonFilter.PersonInfo(authorName, authorEmail, PersonFilter.PersonType.AUTHOR);
            IFilter authorFilter = new PersonFilter(getProject(), author, blameResults);
            FilterUtil.appendFilter(authorFilter);
        }
    }
}
