package io.github.devsecops.engine.domain.resolver.strategy;


import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import io.github.devsecops.engine.core.Log;
import lombok.RequiredArgsConstructor;

import static org.apache.commons.lang3.StringUtils.isNotBlank;

@RequiredArgsConstructor
public class Resolver {

    private final ResolverStrategy propertiesResolver;
    private final ResolverStrategy environmentResolver;
    private final ResolverStrategy vaultStrategyResolver;
    private final Log logger;

    private LoadingCache<String, String> values = CacheBuilder.newBuilder().build(new CacheLoader<String, String>() {
        @Override
        public String load(String propertyName) throws Exception {
            return timeConsumingResolution(propertyName);
        }
    });

    public String resolve(String propertyName) {
        try {
            return values.get(propertyName);
        } catch (Exception e) {
            logger.warn(String.format(">> Property: %s was not resolved, return the key name instead", propertyName));
            return propertyName;
        }
    }

    private String timeConsumingResolution(String propertyName) {
        String propertyValue = propertyName;
        logger.info(">> First time resolving property " + propertyName);
        try {
            propertyValue = propertiesResolver.resolve(propertyName).orElseGet(() -> environmentResolver.resolve(propertyName).get());
            if (isVaultProperty(propertyValue)) {
                return vaultStrategyResolver.resolve(propertyValue).orElse(propertyValue);
            }
        } catch (Exception e) {
            logger.error(String.format(">> Property: %s was not resolved, return the key name instead", propertyName));
            throw e;
        }
        return propertyValue;
    }

    private boolean isVaultProperty(String property) {
        return isNotBlank(property) && property.startsWith("#[vault:");
    }

}
