package io.github.devsecops.engine.domain.sonar.commands;

import io.github.devsecops.engine.core.contract.Command;
import io.github.devsecops.engine.core.exceptions.AbortPipelineException;
import io.github.devsecops.engine.domain.resolver.strategy.Resolver;
import io.github.devsecops.engine.domain.sonar.api.SonarMeasuresApi;
import io.github.devsecops.engine.domain.sonar.model.Metric;
import io.github.devsecops.engine.domain.sonar.model.MetricReport;
import io.github.devsecops.engine.domain.sonar.model.SonarVariables;
import io.github.devsecops.engine.domain.sonar.utils.SonarMeasureValidator;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import org.sonar.wsclient.SonarClient;
import org.springframework.stereotype.Component;

import java.util.Map;

import static java.util.Arrays.asList;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toMap;

@Component
@Slf4j
public class SonarCheckQualityCommand implements Command {

    @Setter
    private Resolver resolver;

    @Override
    public void execute() throws Exception {
        final SonarClient client = SonarClient.builder()
                .url(resolver.resolve(SonarVariables.SONAR_URL.getName()))
                .login(resolver.resolve(SonarVariables.SONAR_LOGIN.getName()))
                .password(resolver.resolve(SonarVariables.SONAR_PASSWORD.getName()))
                .build();
        final SonarMeasuresApi sonarMeasuresApi = new SonarMeasuresApi(client);
        final String component = resolver.resolve(SonarVariables.SONAR_COMPONENT.getName());
        final Map<Metric, Integer> measures = sonarMeasuresApi.calculate(component);
        final boolean isValid = validate(measures);
        if (!isValid) {
            throw new AbortPipelineException("Quality Check failed");
        }
    }

    @Override
    public void rollback() throws Exception {
        throw new UnsupportedOperationException("No rollback is specified for sonar check quality");
    }

    private boolean validate(Map<Metric, Integer> measures) {
        final Map<Metric, Integer> targets = buildMetricTarget();
        final SonarMeasureValidator validator = new SonarMeasureValidator(targets);
        final Map<Metric, MetricReport> validations = validator.perform(measures);
        validations.entrySet().forEach(entry -> log.info(String.format("[%s] %s", entry.getKey().getName(), entry.getValue())));
        return validations.entrySet()
                .stream()
                .map(Map.Entry::getValue)
                .allMatch(MetricReport::isSuccess);
    }

    private Map<Metric, Integer> buildMetricTarget() {
        return asList(Metric.values())
                .parallelStream()
                .collect(toMap(identity(), this::getTarget));
    }

    private Integer getTarget(Metric metric) {
        final SonarVariables property = metric.getTargetValueProperty();
        final String value = resolver.resolve(property.getName());
        return Integer.valueOf(value.trim());
    }

}
