package ch.framedev.javamongodbutils;

import ch.framedev.simplejsonutils.JsonParser;
import com.mongodb.client.FindIterable;
import com.mongodb.client.MongoCollection;
import com.mongodb.client.model.InsertOneOptions;
import com.mongodb.client.result.UpdateResult;
import org.bson.Document;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.function.Consumer;

@SuppressWarnings("unused")
public class BackendMongoDBManager {

    private final MongoDBManager mongoDBManager;

    private final ExecutorService executor = Executors.newFixedThreadPool(10);

    public BackendMongoDBManager(MongoDBManager mongoDBManager) {
        this.mongoDBManager = mongoDBManager;
    }

    public MongoDBManager getMongoManager() {
        return mongoDBManager;
    }

    /**
     * Creating the Document of the User
     *
     * @param where      the Where
     * @param dataWhere  the Data Where
     * @param collection Collection in the Database
     */
    public void createData(String where, Object dataWhere, HashMap<String, Object> data, String collection) {
        if (existsCollection(collection)) {
            MongoCollection<Document> collections = getCollection(collection);
            Document result = collections.find(new Document(where, dataWhere)).first();
            if (result == null) {
                Document dc = (new Document(where, dataWhere));
                dc.putAll(data);
                collections.insertOne(dc, (new InsertOneOptions()).bypassDocumentValidation(false));
            }
        } else {
            mongoDBManager.getDatabase().createCollection(collection);
            MongoCollection<Document> collections = getCollection(collection);
            Document result = collections.find(new Document(where, dataWhere)).first();
            if (result == null) {
                Document dc = (new Document(where, dataWhere));
                dc.putAll(data);
                collections.insertOne(dc, (new InsertOneOptions()).bypassDocumentValidation(false));
            }
        }
    }

    /**
     * Asynchronously creates a new document in the specified collection based on the given conditions.
     * If the document does not exist, it is inserted into the collection.
     *
     * @param where      The field to check in the collection.
     * @param dataWhere  The value to match in the 'where' field.
     * @param data       The key-value pairs to be included in the new document.
     * @param collection The name of the collection in which to create the document.
     * @param callback   The callback to be invoked when the operation is complete.
     */
    public void createDataAsync(String where, Object dataWhere, HashMap<String, Object> data, String collection, Callback<Void> callback) {
        executor.execute(() -> {
            existsCollectionAsync(collection, new Callback<Boolean>() {

                @Override
                public void onResult(Boolean resultBoolean) {
                    if (resultBoolean) {
                        MongoCollection<Document> collections = getCollection(collection);
                        Document result = collections.find(new Document(where, dataWhere)).first();
                        if (result == null) {
                            Document dc = (new Document(where, dataWhere));
                            dc.putAll(data);
                            collections.insertOne(dc, (new InsertOneOptions()).bypassDocumentValidation(false));
                        }
                    } else {
                        mongoDBManager.getDatabase().createCollection(collection);
                        MongoCollection<Document> collections = getCollection(collection);
                        Document result = collections.find(new Document(where, dataWhere)).first();
                        if (result == null) {
                            Document dc = (new Document(where, dataWhere));
                            dc.putAll(data);
                            collections.insertOne(dc, (new InsertOneOptions()).bypassDocumentValidation(false));
                        }
                    }
                }

                @Override
                public void onError(Throwable throwable) {
                    throwable.printStackTrace();
                }
            });
        });
    }

    /**
     * Creates a new document in the specified collection if a document with the given 'where' and 'dataWhere' does not exist.
     *
     * @param where      The field to check in the collection.
     * @param dataWhere  The value to match in the 'where' field.
     * @param object     The object to be converted to a JSON string and inserted into the collection.
     * @param collection The name of the collection in which to insert the document.
     */
    public void createData(String where, Object dataWhere, Object object, String collection) {
        if (existsCollection(collection)) {
            MongoCollection<Document> collections = getCollection(collection);
            Document result = collections.find(new Document(where, dataWhere)).first();
            if (result == null) {
                Document document = Document.parse(new JsonParser().serializeObject(object));
                collections.insertOne(document, (new InsertOneOptions()).bypassDocumentValidation(false));
            }
        } else {
            mongoDBManager.getDatabase().createCollection(collection);
            MongoCollection<Document> collections = getCollection(collection);
            Document result = collections.find(new Document(where, dataWhere)).first();
            if (result == null) {
                Document document = Document.parse(new JsonParser().serializeObject(object));
                collections.insertOne(document, (new InsertOneOptions()).bypassDocumentValidation(false));
            }
        }
    }

    /**
     * Asynchronously creates a new document in the specified collection based on the given conditions.
     *
     * @param where      The field to check in the collection.
     * @param dataWhere  The value to match in the specified field.
     * @param object     The object to be converted into a JSON string and inserted as a new document.
     * @param collection The name of the collection in which to insert the new document.
     * @param callback   The callback to be invoked when the operation is complete.
     */
    public void createDataAsync(String where, Object dataWhere, Object object, String collection, Callback<Void> callback) {
        executor.execute(() -> {
            existsCollectionAsync(collection, new Callback<Boolean>() {
                @Override
                public void onResult(Boolean aBoolean) {
                    if (aBoolean) {
                        MongoCollection<Document> collections = getCollection(collection);
                        Document result = collections.find(new Document(where, dataWhere)).first();
                        if (result == null) {
                            Document document = Document.parse(new JsonParser().serializeObject(object));
                            collections.insertOne(document, (new InsertOneOptions()).bypassDocumentValidation(false));
                        }
                    } else {
                        mongoDBManager.getDatabase().createCollection(collection);
                        MongoCollection<Document> collections = getCollection(collection);
                        Document result = collections.find(new Document(where, dataWhere)).first();
                        if (result == null) {
                            Document document = Document.parse(new JsonParser().serializeObject(object));
                            collections.insertOne(document, (new InsertOneOptions()).bypassDocumentValidation(false));
                        }
                    }
                }

                @Override
                public void onError(Throwable throwable) {
                    throwable.printStackTrace();
                }
            });
        });
    }

    /**
     * Creates a new document in the specified collection based on the given JSON text.
     *
     * @param where      The field to check in the collection.
     * @param dataWhere  The value to match in the 'where' field.
     * @param jsonText   The JSON text representing the new document to be created.
     * @param collection The name of the collection in which to create the document.
     * @throws com.mongodb.MongoException If an error occurs while creating the document.
     */
    public void createData(String where, Object dataWhere, String jsonText, String collection) {
        if (existsCollection(collection)) {
            MongoCollection<Document> collections = getCollection(collection);
            Document result = collections.find(new Document(where, dataWhere)).first();
            if (result == null) {
                Document document = Document.parse(jsonText);
                collections.insertOne(document, (new InsertOneOptions()).bypassDocumentValidation(false));
            }
        } else {
            mongoDBManager.getDatabase().createCollection(collection);
            MongoCollection<Document> collections = getCollection(collection);
            Document result = collections.find(new Document(where, dataWhere)).first();
            if (result == null) {
                Document document = Document.parse(jsonText);
                collections.insertOne(document, (new InsertOneOptions()).bypassDocumentValidation(false));
            }
        }
    }

    /**
     * Inserts multiple key-value pairs into a document in the specified collection.
     *
     * @param where      The field to match in the collection.
     * @param data       The value to match in the collection.
     * @param newKeys    The list of new keys to insert into the document.
     * @param newValues  The list of new values to insert into the document.
     * @param collection The name of the collection in the MongoDB database.
     */
    public void createDataAsync(String where, Object dataWhere, String jsonText, String collection, Callback<Void> callback) {
        executor.execute(() -> {
            existsCollectionAsync(collection, new Callback<Boolean>() {
                @Override
                public void onResult(Boolean aBoolean) {
                    if (aBoolean) {
                        MongoCollection<Document> collections = getCollection(collection);
                        Document result = collections.find(new Document(where, dataWhere)).first();
                        if (result == null) {
                            Document document = Document.parse(jsonText);
                            collections.insertOne(document, (new InsertOneOptions()).bypassDocumentValidation(false));
                        }
                    } else {
                        mongoDBManager.getDatabase().createCollection(collection);
                        MongoCollection<Document> collections = getCollection(collection);
                        Document result = collections.find(new Document(where, dataWhere)).first();
                        if (result == null) {
                            Document document = Document.parse(jsonText);
                            collections.insertOne(document, (new InsertOneOptions()).bypassDocumentValidation(false));
                        }
                    }
                }

                @Override
                public void onError(Throwable throwable) {
                    throwable.printStackTrace();
                }
            });
        });
    }

    /**
     * @param where      from the Database Document
     * @param data       Data in where
     * @param selected   the Selected key in your Database
     * @param collection the Collection in your Database
     * @return data from Database
     */
    public Object getObject(String where, Object data, String selected, String collection) {
        if (existsCollection(collection)) {
            MongoCollection<Document> collections = getCollection(collection);
            Document document = collections.find(new Document(where, data)).first();
            if (document != null) {
                return document.get(selected);
            }
        }
        return null;
    }

    /**
     * Retrieves an object from the specified collection based on the given where condition and selected key.
     *
     * @param where      The condition to filter documents in the collection.
     * @param data       The value to match in the where condition.
     * @param selected   The key in the document to retrieve the value from.
     * @param collection The name of the collection to search in.
     * @param callback   The callback to handle the result.
     */
    public void getObjectAsync(String where, Object data, String selected, String collection, Callback<Object> callback) {
        executor.execute(() -> {
            existsCollectionAsync(collection, new Callback<Boolean>() {
                @Override
                public void onResult(Boolean result) {
                    if (result) {
                        MongoCollection<Document> collections = getCollection(collection);
                        Document document = collections.find(new Document(where, data)).first();
                        if (document != null) {
                            callback.onResult(document.get(selected));
                        } else {
                            callback.onResult(null);
                        }
                    }
                }

                @Override
                public void onError(Throwable throwable) {
                    throwable.printStackTrace();
                }
            });
        });
    }

    /**
     * Retrieves an object from the specified collection based on the given where condition and selected key.
     *
     * @param where      The condition to filter documents in the collection.
     * @param data       The value to match in the where condition.
     * @param selected   The key in the document to retrieve the value from.
     * @param collection The name of the collection to search in.
     * @param type       The class type of the object to be returned.
     * @return The object retrieved from the database, or null if no matching document is found.
     */
    public <T> T getObject(String where, Object data, String selected, String collection, Class<T> type) {
        if (existsCollection(collection)) {
            MongoCollection<Document> collections = getCollection(collection);
            Document document = collections.find(new Document(where, data)).first();
            if (document != null) {
                return document.get(selected, type);
            }
        }
        return null;
    }

    /**
     * Retrieves an object from the specified collection based on the given where condition and selected key.
     *
     * @param where      The condition to filter documents in the collection.
     * @param data       The value to match in the where condition.
     * @param selected   The key in the document to retrieve the value from.
     * @param collection The name of the collection to search in.
     * @param type       The class type of the object to be returned.
     * @param callback   The callback to handle the result.
     */
    public <T> void getObjectAsync(String where, Object data, String selected, String collection, Class<T> type, Callback<T> callback) {
        executor.execute(() -> {
            existsCollectionAsync(collection, new Callback<Boolean>() {
                @Override
                public void onResult(Boolean result) {
                    if (result) {
                        MongoCollection<Document> collections = getCollection(collection);
                        Document document = collections.find(new Document(where, data)).first();
                        if (document != null) {
                            callback.onResult(document.get(selected, type));
                        } else {
                            callback.onResult(null);
                        }
                    }
                }

                @Override
                public void onError(Throwable throwable) {
                    throwable.printStackTrace();
                }
            });
        });
    }

    /**
     * Retrieves a document from the specified collection based on the given where condition.
     *
     * @param where      The condition to filter documents in the collection.
     * @param data       The value to match in the where condition.
     * @param collection The name of the collection to search in.
     * @return The first document that matches the where condition, or null if no matching document is found.
     */
    public Document getDocument(String where, Object data, String collection) {
        if (existsCollection(collection)) {
            MongoCollection<Document> collections = getCollection(collection);
            return collections.find(new Document(where, data)).first();
        }
        return null;
    }

    /**
     * Retrieves a document from the specified collection based on the given where condition.
     *
     * @param where      The condition to filter documents in the collection.
     * @param data       The value to match in the where condition.
     * @param collection The name of the collection to search in.
     * @param callback   The callback to handle the result.
     *                   <p>
     *                   The function executes an asynchronous task to check if the collection exists.
     *                   If the collection exists, it retrieves the first document that matches the where condition
     *                   and passes it to the callback. If the collection does not exist, it prints the error.
     */
    public void getDocumentAsync(String where, Object data, String collection, Callback<Document> callback) {
        executor.execute(() -> {
            existsCollectionAsync(collection, new Callback<Boolean>() {
                @Override
                public void onResult(Boolean result) {
                    if (result) {
                        MongoCollection<Document> collections = getCollection(collection);
                        callback.onResult(collections.find(new Document(where, data)).first());
                    }
                }

                @Override
                public void onError(Throwable throwable) {
                    throwable.printStackTrace();
                }
            });
        });
    }

    /**
     * Retrieves a document from the specified collection based on the given where condition and additional condition.
     *
     * @param where      The condition to filter documents in the collection.
     * @param data       The value to match in the where condition.
     * @param andData    The additional condition to filter documents in the collection.
     * @param andValue   The value to match in the additional condition.
     * @param collection The name of the collection to search in.
     * @return The first document that matches the where and additional conditions, or null if no matching document is found.
     */
    public Document getDocument(String where, Object data, String andData, Object andValue, String collection) {
        if (existsCollection(collection)) {
            MongoCollection<Document> collections = getCollection(collection);
            Document filter = new Document(where, data);
            filter.append(andData, andValue);
            return collections.find(filter).first();
        }
        return null;
    }

    /**
     * Retrieves a document from the specified collection based on the given where condition and additional condition.
     *
     * @param where      The condition to filter documents in the collection.
     * @param data       The value to match in the where condition.
     * @param andData    The additional condition to filter documents in the collection.
     * @param andValue   The value to match in the additional condition.
     * @param collection The name of the collection to search in.
     * @param callback   The callback to handle the result.
     *                   <p>
     *                   The function executes an asynchronous task to check if the collection exists.
     *                   If the collection exists, it retrieves the first document that matches the where and additional conditions
     *                   and passes it to the callback. If the collection does not exist, it prints the error.
     */
    public void getDocumentAsync(String where, Object data, String andData, Object andValue, String collection, Callback<Document> callback) {
        executor.execute(() -> {
            existsCollectionAsync(collection, new Callback<Boolean>() {
                @Override
                public void onResult(Boolean result) {
                    if (result) {
                        MongoCollection<Document> collections = getCollection(collection);
                        Document filter = new Document(where, data);
                        filter.append(andData, andValue);
                        callback.onResult(collections.find(filter).first());
                    }
                }

                @Override
                public void onError(Throwable throwable) {
                    throwable.printStackTrace();
                }
            });
        });
    }

    /**
     * Updates a specific field in a document that matches the given where condition in the specified collection.
     *
     * @param where        The field to match in the where condition.
     * @param data         The value to match in the where condition.
     * @param selected     The field to update in the document.
     * @param dataSelected The new value for the selected field.
     * @param collection   The name of the collection.
     *                     <p>
     *                     If the collection exists and a document matching the where condition is found,
     *                     the selected field is updated with the new value. If the document does not exist,
     *                     a new document is created with the selected field set to the new value.
     *                     If the collection does not exist, it is created and the document is inserted with the selected field set to the new value.
     */
    public void updateData(String where, Object data, String selected, Object dataSelected, String collection) {
        if (existsCollection(collection)) {
            MongoCollection<Document> collections = getCollection(collection);
            Document document = collections.find(new Document(where, data)).first();
            if (document != null) {
                Document document1 = new Document(selected, dataSelected);
                Document document2 = new Document("$set", document1);
                if (document.get(where) != null) {
                    collections.updateOne(document, document2);
                } else {
                    document.put(selected, dataSelected);
                    collections.updateOne(Objects.requireNonNull(collections.find(new Document(where, data)).first()), document);
                }
            }
        } else {
            mongoDBManager.getDatabase().createCollection(collection);
            MongoCollection<Document> collections = getCollection(collection);
            Document document = collections.find(new Document(where, data)).first();
            if (document != null) {
                Document filter = new Document(selected, dataSelected);
                Document query = new Document("$set", filter);
                collections.updateOne(document, query);
            }
        }
    }

    /**
     * Updates a specific field in a document that matches the given where condition in the specified collection.
     *
     * @param where        The field to match in the where condition.
     * @param data         The value to match in the where condition.
     * @param selected     The field to update in the document.
     * @param dataSelected The new value for the selected field.
     * @param collection   The name of the collection.
     *                     <p>
     *                     If the collection exists and a document matching the where condition is found,
     *                     the selected field is updated with the new value. If the document does not exist,
     *                     a new document is created with the selected field set to the new value.
     *                     If the collection does not exist, it is created and the document is inserted with the selected field set to the new value.
     */
    public void updateDataAsync(String where, Object data, String selected, Object dataSelected, String collection, Callback<Void> callback) {
        executor.execute(() -> {
            existsCollectionAsync(collection, new Callback<Boolean>() {
                @Override
                public void onResult(Boolean result) {
                    if (result) {
                        MongoCollection<Document> collections = getCollection(collection);
                        Document document = collections.find(new Document(where, data)).first();
                        if (document != null) {
                            Document document1 = new Document(selected, dataSelected);
                            Document document2 = new Document("$set", document1);
                            if (document.get(where) != null) {
                                collections.updateOne(document, document2);
                            } else {
                                document.put(selected, dataSelected);
                                collections.updateOne(Objects.requireNonNull(collections.find(new Document(where, data)).first()), document);
                            }
                        }
                    } else {
                        mongoDBManager.getDatabase().createCollection(collection);
                        MongoCollection<Document> collections = getCollection(collection);
                        Document document = collections.find(new Document(where, data)).first();
                        if (document != null) {
                            Document filter = new Document(selected, dataSelected);
                            Document query = new Document("$set", filter);
                            collections.updateOne(document, query);
                        }
                    }
                }

                @Override
                public void onError(Throwable throwable) {
                    throwable.printStackTrace();
                }
            });
        });
    }

    /**
     * Updates a specific field in a document that matches the given where condition in the specified collection.
     *
     * @param where      The field to match in the where condition.
     * @param whereData  The value to match in the where condition.
     * @param newData    A HashMap containing the new values to update in the document.
     * @param collection The name of the collection.
     *                   <p>
     *                   If the collection exists and a document matching the where condition is found,
     *                   the document is updated with the new values from the newData HashMap.
     *                   If the document does not exist, no action is taken.
     *                   If the collection does not exist, no action is taken.
     */
    public void updateAll(String where, Object whereData, HashMap<String, Object> newData, String collection) {
        if (existsCollection(collection)) {
            MongoCollection<Document> collections = getCollection(collection);
            Document document = collections.find(new Document(where, whereData)).first();
            if (document != null) {
                if (document.get(where) != null) {
                    Document doc = Document.parse(new JsonParser().serializeObject(newData));
                    collections.replaceOne(document, doc);
                }
            }
        }
    }

    /**
     * Updates a specific field in a document that matches the given where condition in the specified collection.
     *
     * @param where      The field to match in the where condition.
     * @param whereData  The value to match in the where condition.
     * @param newData    A HashMap containing the new values to update in the document.
     * @param collection The name of the collection.
     *                   <p>
     *                   If the collection exists and a document matching the where condition is found,
     *                   the document is updated with the new values from the newData HashMap.
     *                   If the document does not exist, no action is taken.
     *                   If the collection does not exist, no action is taken.
     */
    public void updateAllAsync(String where, Object whereData, HashMap<String, Object> newData, String collection, Callback<Void> callback) {
        executor.execute(() -> {
            existsCollectionAsync(collection, new Callback<Boolean>() {
                @Override
                public void onResult(Boolean result) {
                    if (result) {
                        MongoCollection<Document> collections = getCollection(collection);
                        Document document = collections.find(new Document(where, whereData)).first();
                        if (document != null) {
                            if (document.get(where) != null) {
                                Document doc = Document.parse(new JsonParser().serializeObject(newData));
                                collections.replaceOne(document, doc);
                                callback.onResult(null);
                            }
                        }
                    }
                }

                @Override
                public void onError(Throwable throwable) {
                    throwable.printStackTrace();
                }
            });
        });
    }

    /**
     * Updates a specific field in a document that matches the given where condition in the specified collection.
     *
     * @param where      The field to match in the where condition.
     * @param whereData  The value to match in the where condition.
     * @param object     The new object to replace the existing document with.
     * @param collection The name of the collection.
     *                   <p>
     *                   If the collection exists and a document matching the where condition is found,
     *                   the document is updated with the new object. If the document does not exist,
     *                   no action is taken. If the collection does not exist, no action is taken.
     */
    public void updateAll(String where, Object whereData, Object object, String collection) {
        if (existsCollection(collection)) {
            MongoCollection<Document> collections = getCollection(collection);
            Document document = collections.find(new Document(where, whereData)).first();
            if (document != null) {
                if (document.get(where) != null) {
                    Document doc = Document.parse(new JsonParser().serializeObject(object));
                    collections.replaceOne(document, doc);
                }
            }
        }
    }

    /**
     * Updates a specific field in a document that matches the given where condition in the specified collection.
     *
     * @param where      The field to match in the where condition.
     * @param whereData  The value to match in the where condition.
     * @param object     The new object to replace the existing document with.
     * @param collection The name of the collection.
     * @param callback   A callback to handle the result of the asynchronous operation.
     *                   <p>
     *                   If the collection exists and a document matching the where condition is found,
     *                   the document is updated with the new object. If the document does not exist, no action is taken.
     *                   If the collection does not exist, no action is taken.
     */
    public void updateAllAsync(String where, Object whereData, Object object, String collection, Callback<Void> callback) {
        executor.execute(() -> {
            existsCollectionAsync(collection, new Callback<Boolean>() {
                @Override
                public void onResult(Boolean result) {
                    if (result) {
                        MongoCollection<Document> collections = getCollection(collection);
                        Document document = collections.find(new Document(where, whereData)).first();
                        if (document != null) {
                            if (document.get(where) != null) {
                                Document doc = Document.parse(new JsonParser().serializeObject(object));
                                collections.replaceOne(document, doc);
                                callback.onResult(null);
                            }
                        }
                    }
                }

                @Override
                public void onError(Throwable throwable) {
                    throwable.printStackTrace();
                }
            });
        });
    }

    /**
     * Updates a specific field in a document that matches the given where condition in the specified collection.
     *
     * @param where      The field to match in the where condition.
     * @param whereData  The value to match in the where condition.
     * @param jsonText   The JSON text to replace the existing document with.
     * @param collection The name of the collection.
     *                   <p>
     *                   If the collection exists and a document matching the where condition is found,
     *                   the document is updated with the JSON text. If the document does not exist, no action is taken.
     *                   If the collection does not exist, no action is taken.
     */
    public void updateAll(String where, Object whereData, String jsonText, String collection) {
        if (existsCollection(collection)) {
            MongoCollection<Document> collections = getCollection(collection);
            Document document = collections.find(new Document(where, whereData)).first();
            if (document != null) {
                if (document.get(where) != null) {
                    Document doc = Document.parse(jsonText);
                    collections.replaceOne(document, doc);
                }
            }
        }
    }

    /**
     * Updates a specific field in a document that matches the given where condition in the specified collection.
     *
     * @param where      The field to match in the where condition.
     * @param whereData  The value to match in the where condition.
     * @param jsonText   The JSON text to replace the existing document with.
     * @param collection The name of the collection.
     * @param callback   A callback to handle the result of the asynchronous operation.
     *                   <p>
     *                   If the collection exists and a document matching the where condition is found,
     *                   the document is updated with the JSON text. If the document does not exist, no action is taken.
     *                   If the collection does not exist, no action is taken.
     */
    public void updateAllAsync(String where, Object whereData, String jsonText, String collection, Callback<Void> callback) {
        executor.execute(() -> {
            existsCollectionAsync(collection, new Callback<Boolean>() {
                @Override
                public void onResult(Boolean result) {
                    if (result) {
                        MongoCollection<Document> collections = getCollection(collection);
                        Document document = collections.find(new Document(where, whereData)).first();
                        if (document != null) {
                            if (document.get(where) != null) {
                                Document doc = Document.parse(jsonText);
                                collections.replaceOne(document, doc);
                                callback.onResult(null);
                            }
                        }
                    }
                }

                @Override
                public void onError(Throwable throwable) {
                    throwable.printStackTrace();
                }
            });
        });
    }

    /**
     * Checks if a document exists in a specified collection that meets the given conditions.
     *
     * @param where         The field name to match in the documents.
     * @param data          The value to match in the documents.
     * @param whereSelected The field name to check for existence in the matched documents.
     * @param collection    The name of the collection to search in.
     * @return True if a document exists that matches the given conditions, false otherwise.
     */
    public boolean exists(String where, Object data, String whereSelected, String collection) {
        if (existsCollection(collection)) {
            MongoCollection<Document> collections = getCollection(collection);
            Document document = collections.find(new Document(where, data)).first();
            if (document != null) {
                return document.get(whereSelected) != null;
            }
        }
        return false;
    }

    /**
     * Checks if a document exists in a specified collection that meets the given conditions.
     *
     * @param where         The field name to match in the documents.
     * @param data          The value to match in the documents.
     * @param whereSelected The field name to check for existence in the matched documents.
     * @param collection    The name of the collection to search in.
     * @param callback      A callback to handle the result of the asynchronous operation.
     *                      <p>
     *                      The function executes an asynchronous operation to check if a document exists in the specified collection
     *                      that meets the given conditions. It uses the provided callback to return the result.
     */
    public void existsAsync(String where, Object data, String whereSelected, String collection, Callback<Boolean> callback) {
        executor.execute(() -> {
            existsCollectionAsync(collection, new Callback<Boolean>() {
                @Override
                public void onResult(Boolean result) {
                    if (result) {
                        MongoCollection<Document> collections = getCollection(collection);
                        Document document = collections.find(new Document(where, data)).first();
                        if (document != null) {
                            callback.onResult(document.get(whereSelected) != null);
                        } else {
                            callback.onResult(false);
                        }
                    } else {
                        callback.onResult(false);
                    }
                }

                @Override
                public void onError(Throwable throwable) {
                    throwable.printStackTrace();
                }
            });
        });
    }

    /**
     * Checks if a document exists in a specified collection that meets the given conditions.
     *
     * @param where      The field name to match in the documents.
     * @param data       The value to match in the documents.
     * @param collection The name of the collection to search in.
     * @return True if a document exists that matches the given conditions, false otherwise.
     */
    public boolean exists(String where, Object data, String collection) {
        if (existsCollection(collection)) {
            MongoCollection<Document> collections = getCollection(collection);
            Document document = collections.find(new Document(where, data)).first();
            return document != null;
        }
        return false;
    }

    /**
     * Checks if a document exists in a specified collection that meets the given conditions.
     *
     * @param where      The field name to match in the documents.
     * @param data       The value to match in the documents.
     * @param collection The name of the collection to search in.
     * @param callback   A callback to handle the result of the asynchronous operation.
     *                   <p>
     *                   The function executes an asynchronous operation to check if a document exists in the specified collection
     *                   that meets the given conditions. It uses the provided callback to return the result.
     */
    public void existsAsync(String where, Object data, String collection, Callback<Boolean> callback) {
        executor.execute(() -> {
            existsCollectionAsync(collection, new Callback<Boolean>() {
                @Override
                public void onResult(Boolean result) {
                    if (result) {
                        MongoCollection<Document> collections = getCollection(collection);
                        Document document = collections.find(new Document(where, data)).first();
                        callback.onResult(document != null);
                    } else {
                        callback.onResult(false);
                    }
                }

                @Override
                public void onError(Throwable throwable) {
                    throwable.printStackTrace();
                }
            });
        });
    }

    /**
     * Checks if a document exists in a specified collection that meets the given conditions.
     *
     * @param where         The field name to match in the documents.
     * @param data          The value to match in the documents.
     * @param whereSelected The field name to check for equality in the matched documents.
     * @param whereData     The value to compare with the field specified by 'whereSelected'.
     * @param collection    The name of the collection to search in.
     * @return True if a document exists that matches the given conditions and the field specified by 'whereSelected' equals 'whereData', false otherwise.
     */
    public boolean exists(String where, Object data, String whereSelected, Object whereData, String collection) {
        if (existsCollection(collection)) {
            MongoCollection<Document> collections = getCollection(collection);
            Document document = collections.find(new Document(where, data)).first();
            if (document != null) {
                return document.get(whereSelected) == whereData;
            }
        }
        return false;
    }

    /**
     * Checks if a document exists in a specified collection that meets the given conditions and the field specified by 'whereSelected' equals 'whereData'.
     *
     * @param where         The field name to match in the documents.
     * @param data          The value to match in the documents.
     * @param whereSelected The field name to check for equality in the matched documents.
     * @param whereData     The value to compare with the field specified by 'whereSelected'.
     * @param collection    The name of the collection to search in.
     * @param callback      A callback to handle the result of the asynchronous operation.
     *                      <p>
     *                      The function executes an asynchronous operation to check if a document exists in the specified collection
     *                      that meets the given conditions and the field specified by 'whereSelected' equals 'whereData'. It uses the provided callback to return the result.
     */
    public void existsAsync(String where, Object data, String whereSelected, Object whereData, String collection, Callback<Boolean> callback) {
        executor.execute(() -> {
            existsCollectionAsync(collection, new Callback<Boolean>() {
                @Override
                public void onResult(Boolean result) {
                    if (result) {
                        MongoCollection<Document> collections = getCollection(collection);
                        Document document = collections.find(new Document(where, data)).first();
                        if (document != null) {
                            callback.onResult(document.get(whereSelected) == whereData);
                        } else {
                            callback.onResult(false);
                        }
                    } else
                        callback.onResult(false);
                }

                @Override
                public void onError(Throwable throwable) {
                    throwable.printStackTrace();
                }
            });
        });
    }

    /**
     * Inserts or updates a key-value pair in a document within a specified collection.
     * If the document exists, it updates the document with the new key-value pair.
     * If the document doesn't exist or the collection doesn't exist, no action is taken.
     *
     * @param where      The field name to use for finding the document to update.
     * @param data       The value to match in the 'where' field for finding the document.
     * @param newKey     The key of the new field to insert or update.
     * @param newValue   The value of the new field to insert or update.
     * @param collection The name of the MongoDB collection to operate on.
     */
    public void insertData(String where, Object data, String newKey, Object newValue, String collection) {
        if (existsCollection(collection)) {
            MongoCollection<Document> collections = getCollection(collection);
            Document document = collections.find(new Document(where, data)).first();
            if (document != null) {
                collections.updateOne(new Document(where, data),
                        new Document("$set", new Document(newKey, newValue)));
            }
        }
    }

    /**
     * Asynchronously inserts or updates a key-value pair in a document within a specified collection.
     * If the document exists, it updates the document with the new key-value pair.
     * If the document doesn't exist or the collection doesn't exist, no action is taken.
     *
     * @param where      The field name to use for finding the document to update.
     * @param data       The value to match in the 'where' field for finding the document.
     * @param newKey     The key of the new field to insert or update.
     * @param newValue   The value of the new field to insert or update.
     * @param collection The name of the MongoDB collection to operate on.
     * @param callback   A callback to handle the result of the asynchronous operation. The callback's onResult method will be called with null if the operation is successful.
     */
    public void insertDataAsync(String where, Object data, String newKey, Object newValue, String collection, Callback<Void> callback) {
        executor.execute(() -> {
            existsCollectionAsync(collection, new Callback<Boolean>() {
                @Override
                public void onResult(Boolean result) {
                    if (result) {
                        MongoCollection<Document> collections = getCollection(collection);
                        Document document = collections.find(new Document(where, data)).first();
                        if (document != null) {
                            collections.updateOne(new Document(where, data),
                                    new Document("$set", new Document(newKey, newValue)));
                            callback.onResult(null);
                        }
                    }
                }

                @Override
                public void onError(Throwable throwable) {
                    throwable.printStackTrace();
                }
            });
        });
    }

    /**
     * Inserts or updates multiple key-value pairs in a document within a specified collection.
     * If the document exists, it updates the document with the new key-value pairs.
     * If the document doesn't exist or the collection doesn't exist, no action is taken.
     *
     * @param where      The field name to use for finding the document to update.
     * @param data       The value to match in the 'where' field for finding the document.
     * @param newKeys    A list of keys for the new fields to insert or update.
     * @param newValues  A list of values corresponding to the newKeys to insert or update.
     * @param collection The name of the MongoDB collection to operate on.
     */
    public void insertManyData(String where, Object data, List<String> newKeys, List<Object> newValues, String collection) {
        if (existsCollection(collection)) {
            MongoCollection<Document> collections = getCollection(collection);
            Document filter = new Document(where, data);
            Document updateDocument = new Document();

            // Construct update document with new key-value pairs
            for (int i = 0; i < newKeys.size(); i++) {
                String newKey = newKeys.get(i);
                Object newValue = newValues.get(i);
                updateDocument.append(newKey, newValue);
            }

            // Perform update operation
            UpdateResult result = collections.updateOne(filter, new Document("$set", updateDocument));
            if (result.getModifiedCount() > 0) {
                System.out.println("Document updated successfully");
            } else {
                System.out.println("Document not found or no modifications made");
            }
        }
    }

    /**
     * Asynchronously inserts or updates multiple key-value pairs in a document within a specified collection.
     * If the document exists, it updates the document with the new key-value pairs.
     * If the document doesn't exist or the collection doesn't exist, no action is taken.
     *
     * @param where      The field name to use for finding the document to update.
     * @param data       The value to match in the 'where' field for finding the document.
     * @param newKeys    A list of keys for the new fields to insert or update.
     * @param newValues  A list of values corresponding to the newKeys to insert or update.
     * @param collection The name of the MongoDB collection to operate on.
     * @param callback   A callback to handle the result of the asynchronous operation.
     *                   The callback's onResult method will be called with null if the operation is successful.
     */
    public void insertManyDataAsync(String where, Object data, List<String> newKeys, List<Object> newValues, String collection, Callback<Void> callback) {
        executor.execute(() -> {
            existsCollectionAsync(collection, new Callback<Boolean>() {
                @Override
                public void onResult(Boolean result) {
                    if (result) {
                        MongoCollection<Document> collections = getCollection(collection);
                        Document filter = new Document(where, data);
                        Document updateDocument = new Document();

                        // Construct an update document with new key-value pairs
                        for (int i = 0; i < newKeys.size(); i++) {
                            String newKey = newKeys.get(i);
                            Object newValue = newValues.get(i);
                            updateDocument.append(newKey, newValue);
                        }

                        // Perform update operation
                        UpdateResult resultUpdate = collections.updateOne(filter, new Document("$set", updateDocument));
                        if (resultUpdate.getModifiedCount() > 0) {
                            System.out.println("Document updated successfully");
                        } else {
                            System.out.println("Document not found or no modifications made");
                        }
                        callback.onResult(null);
                    }
                }

                @Override
                public void onError(Throwable throwable) {
                    throwable.printStackTrace();
                }
            });
        });
    }

    /**
     * Checks whether a MongoDB collection with the specified name exists.
     * <p>
     * Internally, this method attempts to retrieve a collection reference
     * by calling {@link #getCollection(String)}. If the returned reference
     * is non-{@code null}, it indicates that the collection exists.
     * </p>
     *
     * @param collection the name of the collection to look up.
     * @return {@code true} if a non-{@code null} reference to the collection
     * was retrieved, indicating the collection exists;
     * {@code false} otherwise.
     */
    public boolean existsCollection(String collection) {
        MongoCollection<Document> collections = getCollection(collection);
        return collections != null;
    }

    /**
     * Asynchronously checks whether a MongoDB collection with the specified name exists.
     *
     * <p>This method submits a task to the configured {@code executor} to retrieve
     * the collection reference via {@link #getCollection(String)}. If the collection
     * reference is non-{@code null}, the callback receives {@code true}. Otherwise,
     * the callback receives {@code false}.
     *
     * @param collection the name of the MongoDB collection to check
     * @param callback   the callback to be invoked with the result of the existence check
     *                   (a {@code Boolean} that is {@code true} if the collection exists,
     *                   or {@code false} otherwise)
     */
    public void existsCollectionAsync(String collection, Callback<Boolean> callback) {
        executor.execute(() -> {
            MongoCollection<Document> collections = getCollection(collection);
            callback.onResult(collections != null);
        });
    }

    /**
     * Retrieves a list of values from documents in a specified MongoDB collection.
     * <p>
     * This method:
     * <ul>
     *   <li>Checks if the specified collection exists (using {@link #existsCollection(String)}).</li>
     *   <li>If it exists, retrieves the collection (using {@link #getCollection(String)}).</li>
     *   <li>Performs a query filtering by {@code where = data}.</li>
     *   <li>For each matching document, extracts the value associated with the {@code selected} field.</li>
     *   <li>Adds those values to a list and returns it.</li>
     *   <li>If the collection does not exist, returns {@code null}.</li>
     * </ul>
     *
     * @param where      the name of the field to filter on
     * @param data       the value that the specified field must match
     * @param selected   the field name whose values are to be extracted from each matching document
     * @param collection the name of the MongoDB collection to query
     * @return a {@link List} of values from the {@code selected} field in each matching document,
     * or {@code null} if the collection does not exist
     */
    public List<Object> getList(String where, Object data, String selected, String collection) {
        ArrayList<Object> list = new ArrayList<>();
        if (existsCollection(collection)) {
            MongoCollection<Document> collections = getCollection(collection);
            collections.find(new Document(where, data)).forEach((Consumer<? super Document>) document -> {
                if (document != null) {
                    list.add(document.get(selected));
                }
            });
            return list;
        }
        return null;
    }

    /**
     * Asynchronously retrieves a list of values from documents in a specified MongoDB collection
     * that match the given query condition. The result is returned via the provided callback.
     *
     * <p>This method:
     * <ul>
     *   <li>Submits a task to the internal {@code executor}.</li>
     *   <li>Asynchronously checks if the specified collection exists via
     *       {@link #existsCollectionAsync(String, Callback)}.</li>
     *   <li>If the collection exists, queries the collection for documents where {@code where = data}.</li>
     *   <li>For each matching document, extracts the value of the {@code selected} field
     *       and adds it to a list.</li>
     *   <li>Invokes {@code callback.onResult(list)} if the collection exists,
     *       or {@code callback.onResult(null)} otherwise.</li>
     *   <li>In case of an error, {@code callback.onError(Throwable)} is invoked.</li>
     * </ul>
     *
     * @param where      the field name to filter on
     * @param data       the value that the specified field must match
     * @param selected   the field name whose values will be extracted from each matching document
     * @param collection the name of the MongoDB collection to query
     * @param callback   a callback that will be invoked with the resulting list of extracted values,
     *                   or with {@code null} if the collection does not exist
     */
    public void getListAsync(String where, Object data, String selected, String collection, Callback<List<Object>> callback) {
        executor.execute(() -> {
            existsCollectionAsync(collection, new Callback<Boolean>() {
                @Override
                public void onResult(Boolean result) {
                    if (result) {
                        ArrayList<Object> list = new ArrayList<>();
                        MongoCollection<Document> collections = getCollection(collection);
                        collections.find(new Document(where, data))
                                .forEach((Consumer<? super Document>) document -> {
                                    if (document != null) {
                                        list.add(document.get(selected));
                                    }
                                });
                        callback.onResult(list);
                    } else {
                        callback.onResult(null);
                    }
                }

                @Override
                public void onError(Throwable throwable) {
                    throwable.printStackTrace();
                }
            });
        });
    }

    /**
     * Retrieves all documents from the specified MongoDB collection.
     * <p>
     * This method:
     * <ul>
     *   <li>Checks if the collection exists by calling {@link #existsCollection(String)}.</li>
     *   <li>If it does exist, obtains the collection via {@link #getCollection(String)}
     *       and iterates over all documents returned by {@code collections.find()}.</li>
     *   <li>Adds each document to a list, which is then returned.</li>
     *   <li>If the collection does not exist, an empty list is returned.</li>
     * </ul>
     *
     * @param collection the name of the MongoDB collection to retrieve documents from
     * @return a {@link List} of all {@link Document} objects in the specified collection,
     * or an empty list if the collection does not exist
     */
    public List<Document> getAllDocuments(String collection) {
        List<Document> list = new ArrayList<>();
        if (existsCollection(collection)) {
            MongoCollection<Document> collections = getCollection(collection);
            FindIterable<Document> find = collections.find();
            for (Document document : find) {
                list.add(document);
            }
        }
        return list;
    }

    /**
     * Asynchronously retrieves all documents from the specified MongoDB collection.
     * <p>
     * This method schedules a task on the internal {@code executor} to:
     * <ul>
     *   <li>Check if the collection exists via {@link #existsCollectionAsync(String, Callback)}.</li>
     *   <li>If the collection exists, retrieve it using {@link #getCollection(String)}.</li>
     *   <li>Iterate over all documents returned by {@code collections.find()} and collect them into a list.</li>
     *   <li>Provide the resulting list to {@code callback.onResult(...)}.</li>
     *   <li>If the collection does not exist, the callback is not invoked with a list (no result is returned).</li>
     *   <li>In case of an error, {@code callback.onError(Throwable)} is invoked.</li>
     * </ul>
     *
     * @param collection the name of the MongoDB collection to retrieve documents from
     * @param callback   the callback that will be invoked with the resulting list of documents
     *                   if the collection exists, or will handle the error if something goes wrong
     */
    public void getAllDocumentsAsync(String collection, Callback<List<Document>> callback) {
        executor.execute(() -> {
            existsCollectionAsync(collection, new Callback<Boolean>() {
                @Override
                public void onResult(Boolean result) {
                    if (result) {
                        List<Document> list = new ArrayList<>();
                        MongoCollection<Document> collections = getCollection(collection);
                        FindIterable<Document> find = collections.find();
                        for (Document document : find) {
                            list.add(document);
                        }
                        callback.onResult(list);
                    }
                }

                @Override
                public void onError(Throwable throwable) {
                    throwable.printStackTrace();
                }
            });
        });
    }

    /**
     * Retrieves all documents from the specified MongoDB collection that match the given filter.
     * <p>
     * This method:
     * <ul>
     *   <li>Checks if the collection exists by calling {@link #existsCollection(String)}.</li>
     *   <li>If it does exist, obtains the collection via {@link #getCollection(String)}
     *       and queries it using {@code collections.find(new Document(whereData, whereValue))}.</li>
     *   <li>Adds each matching document to a list, which is then returned.</li>
     *   <li>If the collection does not exist, an empty list is returned.</li>
     * </ul>
     *
     * @param whereData  the name of the field to filter on
     * @param whereValue the value that the specified field must match
     * @param collection the name of the MongoDB collection to query
     * @return a {@link List} of matching {@link Document} objects from the specified collection,
     * or an empty list if the collection does not exist
     */
    public List<Document> getAllDocuments(String whereData, Object whereValue, String collection) {
        List<Document> list = new ArrayList<>();
        if (existsCollection(collection)) {
            MongoCollection<Document> collections = getCollection(collection);
            FindIterable<Document> find = collections.find(new Document(whereData, whereValue));
            for (Document document : find) {
                list.add(document);
            }
        }
        return list;
    }

    /**
     * Asynchronously retrieves all documents from the specified MongoDB collection
     * that match the given filter. The results are returned via the provided callback.
     *
     * <p>This method schedules a task on the internal {@code executor} to:
     * <ul>
     *   <li>Check if the collection exists via
     *       {@link #existsCollectionAsync(String, Callback)}.</li>
     *   <li>If the collection exists, retrieve it using {@link #getCollection(String)}
     *       and filter with {@code new Document(whereData, whereValue)}.</li>
     *   <li>Collect the matching documents into a list and pass it to
     *       {@code callback.onResult(...)}.</li>
     *   <li>If the collection does not exist, the callback is not invoked with any result.</li>
     *   <li>In case of an error, {@code callback.onError(Throwable)} is invoked.</li>
     * </ul>
     *
     * @param whereData  the name of the field to filter on
     * @param whereValue the value that the specified field must match
     * @param collection the name of the MongoDB collection to query
     * @param callback   the callback that will be invoked with the resulting list of documents
     *                   if the collection exists, or to handle any errors that occur
     */
    public void getAllDocumentsAsync(String whereData, Object whereValue, String collection, Callback<List<Document>> callback) {
        executor.execute(() -> {
            existsCollectionAsync(collection, new Callback<Boolean>() {
                @Override
                public void onResult(Boolean result) {
                    if (result) {
                        List<Document> list = new ArrayList<>();
                        MongoCollection<Document> collections = getCollection(collection);
                        FindIterable<Document> find = collections.find(new Document(whereData, whereValue));
                        for (Document document : find) {
                            list.add(document);
                        }
                        callback.onResult(list);
                    }
                }

                @Override
                public void onError(Throwable throwable) {
                    throwable.printStackTrace();
                }
            });
        });
    }

    /**
     * Retrieves and deserializes all documents from the specified MongoDB collection
     * into objects of the given type.
     *
     * <p>This method:
     * <ul>
     *   <li>Checks if the collection exists using {@link #existsCollection(String)}.</li>
     *   <li>If it does exist, fetches the collection via {@link #getCollection(String)}</li>
     *   <li>Iterates over all documents returned by {@code collections.find()},</li>
     *   <li>Converts each document's JSON representation into an instance of the specified
     *       type using {@code JsonUtils.classFromJsonString(document.toJson(), type)}.</li>
     *   <li>Adds each deserialized object to a list, which is then returned.</li>
     *   <li>If the collection does not exist, an empty list is returned.</li>
     * </ul>
     *
     * @param <T>        the generic type parameter representing the class of objects to be deserialized
     * @param collection the name of the MongoDB collection to query
     * @param type       the {@link Class} object corresponding to {@code <T>} for deserialization
     * @return a list of objects of type {@code T} representing all documents in the collection,
     * or an empty list if the collection does not exist
     */
    @SuppressWarnings("unchecked")
    public <T> List<T> getAllObjects(String collection, Class<T> type) {
        List<T> list = new ArrayList<>();
        if (existsCollection(collection)) {
            MongoCollection<Document> collections = getCollection(collection);
            FindIterable<Document> find = collections.find();
            for (Document document : find) {
                list.add((T) new JsonParser().deserializeObject(document.toJson(), type));
            }
        }
        return list;
    }

    /**
     * Asynchronously retrieves and deserializes all documents from the specified MongoDB collection
     * into objects of the given type, then returns them via the provided callback.
     *
     * <p>This method schedules a task on the internal {@code executor} to:
     * <ul>
     *   <li>Check if the collection exists via {@link #existsCollectionAsync(String, Callback)}.</li>
     *   <li>If the collection exists:
     *     <ul>
     *       <li>Obtain the collection via {@link #getCollection(String)}.</li>
     *       <li>Iterate over all documents returned by {@code collections.find()}.</li>
     *       <li>Convert each document's JSON representation into an instance of the specified
     *           type using {@code JsonUtils.classFromJsonString(...)}.</li>
     *       <li>Collect the deserialized objects into a list and pass that list to
     *           {@code callback.onResult(...)}.</li>
     *     </ul>
     *   </li>
     *   <li>If the collection does not exist, no result is returned to the callback
     *       (i.e., {@code callback.onResult(...)} is not invoked).</li>
     *   <li>In case of an error, {@code callback.onError(Throwable)} is invoked.</li>
     * </ul>
     *
     * @param <T>        the generic type parameter representing the class of objects to be deserialized
     * @param collection the name of the MongoDB collection to query
     * @param type       the {@link Class} object corresponding to {@code <T>} for deserialization
     * @param callback   the callback that will be invoked with the resulting list of {@code T} objects
     *                   if the collection exists, or to handle any errors that occur
     */
    public <T> void getAllObjectsAsync(String collection, Class<T> type, Callback<List<T>> callback) {
        executor.execute(() -> {
            existsCollectionAsync(collection, new Callback<Boolean>() {
                @SuppressWarnings("unchecked")
                @Override
                public void onResult(Boolean result) {
                    if (result) {
                        List<T> list = new ArrayList<>();
                        MongoCollection<Document> collections = getCollection(collection);
                        FindIterable<Document> find = collections.find();
                        for (Document document : find) {
                            list.add((T) new JsonParser().deserializeObject(document.toJson(), type));
                        }
                        callback.onResult(list);
                    }
                }

                @Override
                public void onError(Throwable throwable) {
                    throwable.printStackTrace();
                }
            });
        });
    }

    /**
     * Retrieves a list of all collection names from the current MongoDB database.
     *
     * <p>This method gets the list of collection names by calling
     * {@code getDatabase().listCollectionNames()}, then adds each name
     * to a list and returns that list.</p>
     *
     * @return a {@link List} containing the names of all collections
     * in the current MongoDB database
     */
    public List<String> getAllCollections() {
        List<String> collectionNames = new ArrayList<>();
        this.mongoDBManager.getDatabase().listCollectionNames().forEach(collectionNames::add);
        return collectionNames;
    }

    /**
     * Asynchronously retrieves a list of all collection names from the current MongoDB database.
     *
     * <p>This method schedules a task on the internal {@code executor} that:
     * <ul>
     *   <li>Fetches the collection names from the database using
     *       {@code mongoDBManager.getDatabase().listCollectionNames()}.</li>
     *   <li>Adds those names to a list.</li>
     *   <li>Invokes {@code callback.onResult(...)} with the resulting list.</li>
     * </ul>
     *
     * @param callback the callback that will be invoked with the list of all
     *                 collection names in the current MongoDB database
     */
    public void getAllCollectionsAsync(Callback<List<String>> callback) {
        executor.execute(() -> {
            List<String> collectionNames = new ArrayList<>();
            this.mongoDBManager.getDatabase().listCollectionNames().forEach(collectionNames::add);
            callback.onResult(collectionNames);
        });
    }

    /**
     * Retrieves a single document from the specified MongoDB collection, based on a query filter,
     * and deserializes it into an instance of the specified class.
     *
     * <p>This method:
     * <ul>
     *   <li>Checks if the collection exists using {@link #existsCollection(String)}.</li>
     *   <li>If the collection exists, it queries for the first document where {@code where = whereData}.</li>
     *   <li>If a matching document is found, its JSON representation is deserialized into an instance
     *       of {@code class__} using {@code JsonUtils.classFromJsonString(...)}.</li>
     *   <li>Returns the deserialized object, or {@code null} if no matching document is found or
     *       the collection does not exist.</li>
     * </ul>
     *
     * @param <T>        the type of the object to be deserialized
     * @param where      the field name to filter on
     * @param whereData  the value for the filter condition
     * @param collection the name of the MongoDB collection to query
     * @param class__    the {@link Class} object for the desired return type
     * @return an instance of {@code T} corresponding to the first matching document, or {@code null}
     * if no document is found or if the collection does not exist
     */
    @SuppressWarnings("unchecked")
    public <T> T getObjectFromJson(String where, Object whereData, String collection, Class<T> class__) {
        T t = null;
        if (existsCollection(collection)) {
            MongoCollection<Document> collections = getCollection(collection);
            Document document = collections.find(new Document(where, whereData)).first();
            if (document != null) {
                String json = document.toJson();
                t = (T) new JsonParser().deserializeObject(json, class__);
            }
        }
        return t;
    }

    /**
     * Asynchronously retrieves a single document from the specified MongoDB collection,
     * based on the provided filter condition, and deserializes it into an instance of
     * the specified class. The resulting object is returned via the provided callback.
     *
     * <p>This method schedules a task on the internal {@code executor} that:
     * <ul>
     *   <li>Checks if the collection exists using
     *       {@link #existsCollectionAsync(String, Callback)}.</li>
     *   <li>If the collection exists, queries for the first document matching
     *       {@code where = whereData} via {@code collections.find(...).first()}.</li>
     *   <li>Converts the JSON representation of the retrieved document into an instance
     *       of {@code class__} using
     *       {@code new JsonUtils().classFromJsonString(document.toJson(), class__)}.</li>
     *   <li>Invokes {@code callback.onResult(...)} with the resulting object, or
     *       {@code null} if no matching document was found or the collection does not exist.</li>
     *   <li>In case of an error within the asynchronous check,
     *       {@code callback.onError(Throwable)} is triggered.</li>
     * </ul>
     *
     * @param <T>        the type of the object to be deserialized
     * @param where      the field name for the query condition
     * @param whereData  the value to be matched by the {@code where} field
     * @param collection the name of the MongoDB collection to query
     * @param class__    the {@link Class} object representing the desired return type
     * @param callback   the callback that will be invoked with the deserialized object
     *                   or {@code null} if no document is found or the collection does not exist
     */
    public <T> void getObjectFromJsonAsync(String where, Object whereData, String collection, Class<T> class__, Callback<T> callback) {
        executor.execute(() -> {
            existsCollectionAsync(collection, new Callback<Boolean>() {
                @SuppressWarnings("unchecked")
                @Override
                public void onResult(Boolean result) {
                    T t = null;
                    if (result) {
                        MongoCollection<Document> collections = getCollection(collection);
                        Document document = collections.find(new Document(where, whereData)).first();
                        if (document != null) {
                            String json = document.toJson();
                            t = (T) new JsonParser().deserializeObject(json, class__);
                        }
                    }
                    callback.onResult(t);
                }

                @Override
                public void onError(Throwable throwable) {
                    throwable.printStackTrace();
                }
            });
        });
    }

    /**
     * Removes one or more documents from the specified MongoDB collection that match
     * a given query condition.
     *
     * <p>This method:
     * <ul>
     *   <li>Checks if the collection exists by calling {@link #existsCollection(String)}.</li>
     *   <li>If the collection exists, it retrieves the collection via {@link #getCollection(String)}.</li>
     *   <li>Finds the first document where {@code where = whereData}.</li>
     *   <li>If such a document is found, it removes all matching documents via
     *       {@code collections.deleteMany(document)}.</li>
     * </ul>
     *
     * @param where      the field name to filter on
     * @param whereData  the value for the query condition
     * @param collection the name of the MongoDB collection from which documents will be removed
     */
    public void removeDocument(String where, Object whereData, String collection) {
        if (existsCollection(collection)) {
            MongoCollection<Document> collections = getCollection(collection);
            Document document = collections.find(new Document(where, whereData)).first();
            if (document != null) {
                collections.deleteMany(document);
            }
        }
    }

    public MongoDBManager getMongoDBManager() {
        return mongoDBManager;
    }

    /**
     * Asynchronously removes one or more documents from the specified MongoDB collection
     * that match a given query condition. The result is returned via the provided callback.
     *
     * <p>This method schedules a task on the internal {@code executor} to:
     * <ul>
     *   <li>Check if the collection exists via {@link #existsCollectionAsync(String, Callback)}.</li>
     *   <li>If the collection exists, retrieve the first document where {@code where = whereData}.</li>
     *   <li>If such a document is found, remove all matching documents using {@code collections.deleteMany(document)}.</li>
     *   <li>Invoke {@code callback.onResult(null)} upon completion.</li>
     *   <li>Invoke {@code callback.onError(Throwable)} in case of an error.</li>
     * </ul>
     *
     * @param where      the field name to filter on
     * @param whereData  the value for the query condition
     * @param collection the name of the MongoDB collection from which documents will be removed
     * @param callback   the callback that will be invoked once the operation completes;
     *                   {@code onResult(null)} will be called if the collection exists
     *                   (regardless of whether any documents were removed), or
     *                   {@code onError(Throwable)} if an error occurs
     */
    public void removeDocumentAsync(String where, Object whereData, String collection, Callback<Void> callback) {
        executor.execute(() -> {
            existsCollectionAsync(collection, new Callback<Boolean>() {
                @Override
                public void onResult(Boolean result) {
                    if (result) {
                        MongoCollection<Document> collections = getCollection(collection);
                        Document document = collections.find(new Document(where, whereData)).first();
                        if (document != null) {
                            collections.deleteMany(document);
                        }
                        callback.onResult(null);
                    }
                }

                @Override
                public void onError(Throwable throwable) {
                    throwable.printStackTrace();
                }
            });
        });
    }

    /**
     * Removes one or more documents from the specified MongoDB collection
     * that match two query conditions combined with a logical AND.
     *
     * <p>This method:
     * <ul>
     *   <li>Checks if the collection exists via {@link #existsCollection(String)}.</li>
     *   <li>If the collection exists, constructs a filter using
     *       {@code where = whereData} and {@code andData = andValue}.</li>
     *   <li>Finds the first document matching that filter.</li>
     *   <li>If such a document is found, removes all matching documents by
     *       calling {@code collections.deleteMany(document)}.</li>
     * </ul>
     *
     * @param where      the first field name for the query condition
     * @param whereData  the value that the {@code where} field must match
     * @param andData    the second field name for the query condition
     * @param andValue   the value that the {@code andData} field must match
     * @param collection the name of the MongoDB collection from which documents will be removed
     */
    public void removeDocument(String where, Object whereData, String andData, Object andValue, String collection) {
        if (existsCollection(collection)) {
            MongoCollection<Document> collections = getCollection(collection);
            Document filter = new Document(where, whereData);
            filter.append(andData, andValue);
            Document document = collections.find(filter).first();
            if (document != null) {
                collections.deleteMany(document);
            }
        }
    }

    /**
     * Asynchronously removes one or more documents from the specified MongoDB collection
     * that match two query conditions combined with a logical AND, then invokes a callback
     * upon completion.
     *
     * <p>This method schedules a task on the internal {@code executor} to:
     * <ul>
     *   <li>Check if the collection exists via
     *       {@link #existsCollectionAsync(String, Callback)}.</li>
     *   <li>If the collection exists:
     *     <ul>
     *       <li>Construct a filter with {@code where = whereData} and
     *           {@code andData = andValue}.</li>
     *       <li>Find the first document that matches this filter.</li>
     *       <li>If such a document is found, remove all matching documents using
     *           {@code collections.deleteMany(document)}.</li>
     *       <li>Invoke {@code callback.onResult(null)} to signal completion.</li>
     *     </ul>
     *   </li>
     *   <li>In case of an error, {@code callback.onError(Throwable)} is invoked.</li>
     * </ul>
     *
     * @param where      the first field name for the query condition
     * @param whereData  the value that the {@code where} field must match
     * @param andData    the second field name for the query condition
     * @param andValue   the value that the {@code andData} field must match
     * @param collection the name of the MongoDB collection from which documents should be removed
     * @param callback   the callback to be invoked upon completion;
     *                   {@code callback.onResult(null)} is called if the collection exists
     *                   (regardless of whether any documents were removed),
     *                   or {@code callback.onError(Throwable)} if an error occurs
     */
    public void removeDocumentAsync(String where, Object whereData, String andData, Object andValue,
                                    String collection, Callback<Void> callback) {
        executor.execute(() -> {
            existsCollectionAsync(collection, new Callback<Boolean>() {
                @Override
                public void onResult(Boolean result) {
                    if (result) {
                        MongoCollection<Document> collections = getCollection(collection);
                        Document filter = new Document(where, whereData);
                        filter.append(andData, andValue);
                        Document document = collections.find(filter).first();
                        if (document != null) {
                            collections.deleteMany(document);
                        }
                        callback.onResult(null);
                    }
                }

                @Override
                public void onError(Throwable throwable) {
                    throwable.printStackTrace();
                }
            });
        });
    }

    /**
     * Retrieves the specified MongoDB collection from the current database.
     *
     * <p>This method:
     * <ul>
     *   <li>Obtains the current MongoDB database via {@code mongoDBManager.getDatabase()}.</li>
     *   <li>Calls {@code getCollection(collectionName)} on the database to retrieve the desired collection.</li>
     *   <li>Returns a {@link MongoCollection} referencing the specified collection.</li>
     * </ul>
     *
     * @param collectionName the name of the MongoDB collection to retrieve
     * @return a {@link MongoCollection} of {@link Document} referencing the requested collection
     */
    public MongoCollection<Document> getCollection(String collectionName) {
        return this.mongoDBManager.getDatabase().getCollection(collectionName);
    }
}
