package com.liecoder.framework.base

import android.content.Intent
import android.content.pm.PackageManager
import android.net.Uri
import android.os.Bundle
import android.provider.Settings
import android.view.Gravity
import androidx.activity.result.ActivityResultCallback
import androidx.activity.result.ActivityResultLauncher
import androidx.activity.result.contract.ActivityResultContracts
import androidx.appcompat.app.AlertDialog
import androidx.appcompat.app.AppCompatActivity
import androidx.core.app.ActivityCompat
import androidx.core.content.ContextCompat
import com.liecoder.framework.model.Permission
import com.liecoder.framework.toast.TipsToast
import com.liecoder.framework.toast.TipsToast.Errors

/**
 * Author: liecoder
 * Date: 2024/6/21 周五
 * Version: 1.0
 */
abstract class BasePermissionActivity : AppCompatActivity(), IPermissionService,
                                        ActivityResultCallback<Boolean> {

    protected val TAG = this::class.java.simpleName
    private lateinit var requestPermissionLauncher: ActivityResultLauncher<String>
    private var onPermissionGranted: (() -> Unit)? = null
    private var onPermissionDenied: (() -> Unit)? = null
    private var permissionExplanationDialog: AlertDialog? = null

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        requestPermissionLauncher = registerPermissionLauncher()
    }

    override fun registerPermissionLauncher(): ActivityResultLauncher<String> {
        return registerForActivityResult(
            ActivityResultContracts.RequestPermission(),
            this
        )
    }

    final override fun onActivityResult(isGranted: Boolean?) {
        if (isGranted == true) {
            onPermissionGranted?.invoke()
        } else {
            onPermissionDenied?.invoke()
        }
        permissionExplanationDialog?.dismiss()
    }

    override fun withPermission(
        permission: Permission,
        onPermissionGranted: () -> Unit,
        onPermissionDenied: (() -> Unit)?
    ) {
        this.onPermissionGranted = onPermissionGranted
        this.onPermissionDenied = onPermissionDenied
        when {
            ContextCompat.checkSelfPermission(
                this,
                permission.permissionType
            ) == PackageManager.PERMISSION_GRANTED -> onPermissionGranted.invoke()
            ActivityCompat.shouldShowRequestPermissionRationale(
                this,
                permission.permissionType
            )                                      -> showRequestPermissionRationaleDialog(
                permission
            )
            else                                   -> {
                requestPermissionLauncher.launch(permission.permissionType)
                explainAccessToMoreSensitiveInformation(permission)
            }
        }
    }

    override fun showRequestPermissionRationaleDialog(
        permission: Permission
    ) {
        AlertDialog.Builder(this@BasePermissionActivity)
            .setMessage("你已禁止授予应用 ${permission.permissionName} 权限，可能会造成功能不可用，如需使用请到设置里授予权限")
            .setNegativeButton(
                "取消"
            ) { dialog, _ ->
                dialog.dismiss()
            }.setPositiveButton("前往授权") { dialog, _ ->
                dialog.dismiss()
                grantingPermissions()
            }.show()
    }

    override fun grantingPermissions() {
        val intent = Intent()
        intent.action = Settings.ACTION_APPLICATION_DETAILS_SETTINGS
        val uri = Uri.fromParts(
            "package",
            packageName,
            null
        )
        intent.data = uri
        if (intent.resolveActivity(packageManager) != null) {
            startActivity(intent)
        } else {
            Errors.show("无法打开应用信息页面")
        }
    }

    override fun getExplanationForSensitiveInformation(permission: Permission): String? {
        return when (permission) {
            Permission.CAMERA                -> "应用需要申请相机权限，以便您可以正常使用扫码，实名认证，拍摄和上传照片等服务"
            Permission.READ_PHONE_STATE      -> "为了信息推送和账号安全，我们会申请系统设备权限收集设备信息、日志信息等"
            Permission.READ_CONTACTS         -> "用用需要申请通讯录权限，用以匹配或添加联系人（未经您同意，我们不会从第三方获取、共享或对外提供您的信息）"
            Permission.READ_EXTERNAL_STORAGE -> "应用需要访问您的存储权限，以便您可以正常使用文件导入、签字、发送、下载、实名认证等服务"
            else                             -> null
        }
    }

    override fun explainAccessToMoreSensitiveInformation(
        permission: Permission
    ) {
        val explanation = getExplanationForSensitiveInformation(permission) ?: return
        permissionExplanationDialog =
            AlertDialog.Builder(this@BasePermissionActivity).setTitle("权限使用说明")
                .setMessage(explanation).create().apply {
                    setOnShowListener {
                        permissionExplanationDialog?.window?.setGravity(Gravity.TOP)
                    }
                }
        permissionExplanationDialog?.show()
    }

}