package com.liecoder.framework.ktx

import android.content.res.Resources
import android.util.TypedValue


/**
 * Converts an integer DP (Density-independent Pixels) value to a float pixel value on the current device.
 *
 * This property delegate provides a convenient way to convert an integer DP value to a float pixel value, taking into account the screen density.
 * It uses the [TypedValue.applyDimension] method with [TypedValue.COMPLEX_UNIT_DIP] to perform the conversion based on the system's display metrics.
 *
 * Usage example:
 * ```kotlin
 * val dimensionInPxF: Float = 16.dpFloat // Converts 16dp to pixels as a Float
 * ```
 *
 * @return The converted float pixel value.
 */
inline val Int.dpFloat: Float
    get() = TypedValue.applyDimension(
        TypedValue.COMPLEX_UNIT_DIP,
        this.toFloat(),
        Resources.getSystem().displayMetrics
    )

/**
 * Converts an integer SP (Scale-independent Pixels) value to a float pixel value on the current device.
 *
 * This property delegate provides a convenient way to convert an integer SP value to a float pixel value, taking into account the user's font size preference.
 * It uses the [TypedValue.applyDimension] method with [TypedValue.COMPLEX_UNIT_SP] to perform the conversion based on the system's display metrics.
 *
 * Usage example:
 * ```kotlin
 * val dimensionInPxF: Float = 14.spFloat // Converts 14sp to pixels as a Float
 * ```
 *
 * @return The converted float pixel value.
 */
inline val Int.spFloat: Float
    get() = TypedValue.applyDimension(
        TypedValue.COMPLEX_UNIT_SP,
        this.toFloat(),
        Resources.getSystem().displayMetrics
    )

/**
 * 将整数（Int）的 DP（Density-independent Pixels）值转换为整型像素值（Int）。
 *
 * 此属性委托提供了一种便捷的方式来获取当前设备上 Int 类型 DP 值对应的整型像素值。
 * 使用 TypedValue 类的 [applyDimension] 方法，先将 Int 值转换为 Float，然后根据系统资源的显示度量来转换尺寸。
 *
 * 使用示例：
 * ```kotlin
 * val dimensionInPx: Int = 16.dpInt // 将16dp的整数转换为像素的Int表示
 * ```
 *
 * @return 返回转换后的整型像素值。
 */
inline val Int.dpInt: Int
    get() = TypedValue.applyDimension(
        TypedValue.COMPLEX_UNIT_DIP,
        this.toFloat(),
        Resources.getSystem().displayMetrics
    ).toInt()

/**
 * 将整数（Int）的 SP（Scale-independent Pixels）值转换为整型像素值（Int）。
 *
 * 此属性委托提供了一种便捷的方式来获取当前设备上 Int 类型 SP 值对应的整型像素值。
 * 使用 TypedValue 类的 [applyDimension] 方法，先将 Int 值转换为 Float，然后根据系统资源的显示度量来转换尺寸。
 *
 * 使用示例：
 * ```kotlin
 * val dimensionInPx: Int = 14.spInt // 将14sp的整数转换为像素的Int表示
 * ```
 *
 * @return 返回转换后的整型像素值。
 */
inline val Int.spInt: Int
    get() = TypedValue.applyDimension(
        TypedValue.COMPLEX_UNIT_SP,
        this.toFloat(),
        Resources.getSystem().displayMetrics
    ).toInt()

/**
 * 将浮点数（Float）的 DP（Density-independent Pixels）值转换为浮点型像素值（Float）。
 *
 * 此属性委托提供了一种便捷的方式来获取当前设备上 Float 类型 DP 值对应的浮点型像素值。
 * 使用 TypedValue 类的 [applyDimension] 方法，根据系统资源的显示度量来转换尺寸。
 *
 * 使用示例：
 * ```kotlin
 * val dimensionInPxF: Float = 16f.dpFloat // 将16dp的浮点数转换为像素的Float表示
 * ```
 *
 * @return 返回转换后的浮点型像素值。
 */
inline val Float.dpFloat: Float
    get() = TypedValue.applyDimension(
        TypedValue.COMPLEX_UNIT_DIP,
        this,
        Resources.getSystem().displayMetrics
    )

/**
 * Converts a float SP (Scale-independent Pixels) value to a float pixel value on the current device.
 *
 * This property delegate provides a convenient way to convert a float SP value to a float pixel value, taking into account the user's font size preference.
 * It uses the [TypedValue.applyDimension] method with [TypedValue.COMPLEX_UNIT_SP] to perform the conversion based on the system's display metrics.
 *
 * Usage example:
 * ```kotlin
 * val dimensionInPxF: Float = 14f.spFloat // Converts 14.0sp to pixels as a Float
 * ```
 *
 * @return The converted float pixel value.
 */
inline val Float.spFloat: Float
    get() = TypedValue.applyDimension(
        TypedValue.COMPLEX_UNIT_SP,
        this,
        Resources.getSystem().displayMetrics
    )

/**
 * 将浮点数（Float）的 DP（Density-independent Pixels）值转换为整型像素值（Int）。
 *
 * 此属性委托提供了一种便捷的方式来获取当前设备上 Float 类型 DP 值对应的整型像素值。
 * 使用 TypedValue 类的 [applyDimension] 方法，根据系统资源的显示度量来转换尺寸。
 *
 * 使用示例：
 * ```kotlin
 * val dimensionInPx: Int = 16f.dpInt // 将16dp的浮点数转换为像素的Int表示
 * ```
 *
 * @return 返回转换后的整型像素值。
 */
inline val Float.dpInt: Int
    get() = TypedValue.applyDimension(
        TypedValue.COMPLEX_UNIT_DIP,
        this,
        Resources.getSystem().displayMetrics
    ).toInt()

/**
 * 将浮点数（Float）的 SP（Scale-independent Pixels）值转换为整型像素值（Int）。
 *
 * 此属性委托提供了一种便捷的方式来获取当前设备上 Float 类型 SP 值对应的整型像素值。
 * 使用 TypedValue 类的 [applyDimension] 方法，根据系统资源的显示度量来转换尺寸。
 *
 * 使用示例：
 * ```kotlin
 * val dimensionInPx: Int = 14f.spInt // 将14sp的浮点数转换为像素的Int表示
 * ```
 *
 * @return 返回转换后的整型像素值。
 */
inline val Float.spInt: Int
    get() = TypedValue.applyDimension(
        TypedValue.COMPLEX_UNIT_SP,
        this,
        Resources.getSystem().displayMetrics
    ).toInt()