package com.liecoder.framework.ktx

import android.content.Context
import android.content.res.ColorStateList
import android.graphics.BlendMode
import android.graphics.BlendModeColorFilter
import android.graphics.Color
import android.graphics.PorterDuff
import android.graphics.drawable.Drawable
import android.os.Build
import android.view.View
import androidx.annotation.ColorRes
import androidx.annotation.DrawableRes
import androidx.core.content.ContextCompat
import androidx.core.content.res.ResourcesCompat

/**
 * 判断当前应用的布局方向是否为从右到左（RTL）。
 *
 * 从 Android 4.2（API 级别 17，Jelly Bean MR1）开始，可以使用系统的 API 来检测布局方向。
 * 如果当前 API 级别小于 17，则默认返回 `false`。
 *
 * @return 如果布局方向为 RTL，则返回 `true`；否则返回 `false`。
 */
inline val Context.isLayoutRtl: Boolean
    get() = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) false else
        resources.configuration.layoutDirection == View.LAYOUT_DIRECTION_RTL

/**
 * 根据资源 ID 获取与之对应的 Drawable 对象，并确保其兼容性。
 *
 * 此函数使用 ContextCompat 的getDrawable方法来获取 Drawable 对象，这可以确保在不同版本的 Android 上均能正常工作。
 *
 * @param idRes Drawable 资源的 ID。
 * @return 对应于给定资源 ID 的 Drawable 对象，如果没有找到对应的资源，则返回 null。
 */
fun Context.getDrawableCompat(@DrawableRes idRes: Int): Drawable? =
    ContextCompat.getDrawable(this, idRes)

/**
 * 根据颜色资源 ID 获取与之对应的颜色值，并确保其兼容性。
 *
 * 此函数使用 ContextCompat 的 getColor 方法来获取颜色值，这可以确保在不同版本的 Android 上均能正常工作。
 *
 * @param colorRes 颜色资源的 ID。
 * @return 对应于给定资源 ID 的颜色值。
 */
fun Context.getColorCompat(@ColorRes colorRes: Int): Int =
    ContextCompat.getColor(this, colorRes)

/**
 * 根据颜色状态资源 ID 获取与之对应的颜色状态列表（ColorStateList），并确保其兼容性。
 *
 * 此函数使用 ResourcesCompat 的 getColorStateList 方法来获取颜色状态列表，适用于不同版本的 Android。
 * 颜色状态列表允许定义在不同状态下显示不同颜色的 Drawable。
 *
 * @param colorRes 颜色状态资源的 ID。
 * @return 对应于给定资源 ID 的颜色状态列表，如果没有找到对应的资源，则返回 null。
 */
fun Context.getColorStateListCompat(@ColorRes colorRes: Int): ColorStateList? =
    ResourcesCompat.getColorStateList(resources, colorRes, theme)

/**
 * 根据条件为 Drawable 应用或移除颜色滤镜（tint）。
 *
 * 此函数允许开发者根据 [shouldApplyTint] 参数决定是否为 Drawable 应用颜色滤镜。如果 [shouldApplyTint] 为 true，
 * 则使用指定的 [tint] 颜色值应用颜色滤镜；如果为 false，则移除任何已应用的颜色滤镜。
 * 颜色滤镜的应用方式根据 Android 系统版本决定，Android 10（API 级别 29，Q）及以上版本使用 [BlendModeColorFilter]，
 * 低版本使用 [setColorFilter] 方法结合 [PorterDuff.Mode.SRC_IN] 模式。
 *
 * @param shouldApplyTint 是否应用颜色滤镜的布尔值，默认为 true。
 * @param tint 要应用的颜色值，默认为透明（[Color.TRANSPARENT]）。
 * @return 返回 Drawable 实例自身，允许链式调用。
 */
fun Drawable.applyTint(shouldApplyTint: Boolean = true, tint: Int = Color.TRANSPARENT): Drawable {
    // 创建 Drawable 的可修改副本
    if (tint == Color.TRANSPARENT) return this
    mutate()
    if (shouldApplyTint) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
            colorFilter = BlendModeColorFilter(tint, BlendMode.SRC_IN)
        } else {
            setColorFilter(tint, PorterDuff.Mode.SRC_IN)
        }
    } else {
        clearColorFilter()
    }
    return this
}

/**
 * 为 Drawable 设置尺寸，并保持原始宽高比。
 *
 * 此函数允许开发者为 Drawable 设置新的宽度和高度。如果只指定了一个维度（宽度或高度），函数会根据 Drawable 的原始宽高比计算缺失的维度。
 * 如果没有指定有效的宽度或高度（即两者都小于或等于0），则重置为 Drawable 的固有尺寸。
 *
 * @param width 要设置的宽度，或 null 如果不更改宽度。
 * @param height 要设置的高度，或 null 如果不更改高度。
 * @return 返回 Drawable 实例自身，允许链式调用。
 */
fun Drawable.applySize(width: Int?, height: Int?): Drawable {

    val drawableWidth = width?.takeIf { it > 0 } ?: 0
    val drawableHeight = height?.takeIf { it > 0 } ?: 0

    if (drawableWidth <= 0 && drawableHeight <= 0) {
        setBounds(0, 0, intrinsicWidth, intrinsicHeight)
        return this
    }
    if (drawableWidth > 0 && drawableHeight > 0) {
        setBounds(0, 0, drawableWidth, drawableHeight)
        return this
    }
    val drawableWidthIntrinsic = if (intrinsicWidth > 0) intrinsicWidth else drawableWidth
    val drawableHeightIntrinsic = if (intrinsicHeight > 0) intrinsicHeight else drawableHeight

    if (drawableWidth > 0){
        val ratio = drawableHeightIntrinsic.toFloat() / drawableWidthIntrinsic
        setBounds(0, 0, drawableWidth, (drawableWidth * ratio).toInt())
    }else{
        val ratio = drawableWidthIntrinsic.toFloat() / drawableHeightIntrinsic
        setBounds(0, 0, (drawableHeight * ratio).toInt(), drawableHeight)
    }

    return this
}