package com.liecoder.framework.ktx

import android.graphics.Color
import android.graphics.Typeface
import android.text.Spannable
import android.text.SpannableString
import android.text.TextPaint
import android.text.style.AbsoluteSizeSpan
import android.text.style.BackgroundColorSpan
import android.text.style.ClickableSpan
import android.text.style.ForegroundColorSpan
import android.text.style.RelativeSizeSpan
import android.text.style.StrikethroughSpan
import android.text.style.StyleSpan
import android.view.View

/**
 * 将一段文字中指定range的文字改变大小
 * @param range 要改变大小的文字的范围
 * @param scale 缩放值，大于1，则比其他文字大；小于1，则比其他文字小；默认是1.5
 */
fun CharSequence.toSizeSpan(
        range: IntRange,
        scale: Float = 1.5f
): SpannableString {
    return SpannableString(this).apply {
        setSpan(
                RelativeSizeSpan(scale),
                range.first,
                range.last,
                Spannable.SPAN_INCLUSIVE_EXCLUSIVE
        )
    }
}

fun CharSequence.toSizeSpan(
        range: IntRange,
        textSize: Int = 16
): SpannableString {
    return SpannableString(this).apply {
        setSpan(
                AbsoluteSizeSpan(textSize.dpInt),
                range.first,
                range.last,
                Spannable.SPAN_INCLUSIVE_EXCLUSIVE
        )
    }
}

/**
 * 将一段文字中指定range的文字改变前景色
 * @param range 要改变前景色的文字的范围
 * @param color 要改变的颜色，默认是红色
 */
fun CharSequence.toColorSpan(
        range: IntRange,
        color: Int = Color.RED
): SpannableString {
    return SpannableString(this).apply {
        setSpan(
                ForegroundColorSpan(color),
                range.first,
                range.last,
                Spannable.SPAN_INCLUSIVE_EXCLUSIVE
        )
    }
}

/**
 * 将一段文字中指定range的文字改变背景色
 * @param range 要改变背景色的文字的范围
 * @param color 要改变的颜色，默认是红色
 */
fun CharSequence.toBackgroundColorSpan(
        range: IntRange,
        color: Int = Color.RED
): SpannableString {
    return SpannableString(this).apply {
        setSpan(
                BackgroundColorSpan(color),
                range.first,
                range.last,
                Spannable.SPAN_INCLUSIVE_EXCLUSIVE
        )
    }
}

/**
 * 将一段文字中指定range的文字添加删除线
 * @param range 要添加删除线的文字的范围
 */
fun CharSequence.toStrikeThroughtSpan(range: IntRange): SpannableString {
    return SpannableString(this).apply {
        setSpan(
                StrikethroughSpan(),
                range.first,
                range.last,
                Spannable.SPAN_INCLUSIVE_EXCLUSIVE
        )
    }
}

/**
 * 将一段文字中指定range的文字添加颜色和点击事件
 * @param range 目标文字的范围
 */
fun CharSequence.toClickSpan(
        range: IntRange,
        color: Int = Color.RED,
        isUnderlineText: Boolean = false,
        clickAction: () -> Unit
): SpannableString {
    return SpannableString(this).apply {
        val clickableSpan = object : ClickableSpan() {
            override fun onClick(widget: View) {
                clickAction()
            }

            override fun updateDrawState(ds: TextPaint) {
                ds.color = color
                ds.isUnderlineText = isUnderlineText
            }
        }
        setSpan(
                clickableSpan,
                range.first,
                range.last,
                Spannable.SPAN_INCLUSIVE_EXCLUSIVE
        )
    }
}

/**
 * 将一段文字中指定range的文字添加style效果
 * @param range 要添加删除线的文字的范围
 */
fun CharSequence.toStyleSpan(
        range: IntRange,
        style: Int = Typeface.BOLD
): SpannableString {
    return SpannableString(this).apply {
        setSpan(
                StyleSpan(style),
                range.first,
                range.last,
                Spannable.SPAN_INCLUSIVE_EXCLUSIVE
        )
    }
}