package com.liecoder.framework.titlebar

import android.content.res.ColorStateList
import android.graphics.Color
import android.graphics.Typeface
import android.graphics.drawable.ColorDrawable
import android.graphics.drawable.Drawable
import android.os.Build
import android.text.TextUtils.TruncateAt
import android.util.TypedValue.COMPLEX_UNIT_PX
import android.view.Gravity
import android.widget.FrameLayout
import androidx.annotation.DrawableRes
import androidx.annotation.StringRes
import androidx.core.content.res.ResourcesCompat
import com.liecoder.framework.ktx.applyEllipsize
import com.liecoder.framework.ktx.applySize
import com.liecoder.framework.ktx.applyTint
import com.liecoder.framework.ktx.asTypeface
import com.liecoder.framework.ktx.containsContent
import com.liecoder.framework.ktx.e
import com.liecoder.framework.ktx.getAbsoluteGravity
import com.liecoder.framework.ktx.getCompoundDrawableGravity
import com.liecoder.framework.ktx.isLayoutRtl
import com.liecoder.framework.ktx.setCompoundDrawableGravity
import com.liecoder.framework.ktx.setVisible

/**
 * 设置标题栏的标题文本。
 *
 * 使用资源 ID 设置标题栏的标题文本。这重载方法允许您直接传递一个字符串资源 ID。
 *
 * @param titleRes 标题文本的资源 ID。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setTitle(@StringRes titleRes: Int): TitleBar = setTitle(context.getString(titleRes))

/**
 * 设置标题栏的格式化标题文本。
 *
 * 使用资源 ID 设置标题栏的格式化标题文本。此重载方法允许您传递一个字符串资源 ID 和格式化参数。
 *
 * @param titleRes 格式化标题文本的资源 ID。
 * @param formatArgs 格式化参数。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setTitle(@StringRes titleRes: Int, vararg formatArgs: Any): TitleBar =
    setTitle(context.getString(titleRes, *formatArgs))

/**
 * 设置标题栏的标题文本。
 *
 * 使用 CharSequence 设置标题栏的标题文本。此重载方法允许您直接传递 CharSequence 对象。
 *
 * @param text 要设置的标题文本。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setTitle(text: CharSequence?): TitleBar {
    mTitleView.text = text
    return this
}

/**
 * 获取标题栏当前的标题文本。
 *
 * 此函数返回标题栏上设置的标题文本，如果标题文本没有被设置，则可能返回 null。
 *
 * @return 返回标题栏上的标题文本，类型为 CharSequence?。
 */
fun TitleBar.getTitle(): CharSequence? = mTitleView.text

/**
 * 设置标题栏左侧标题文本。
 *
 * 使用资源 ID 设置标题栏左侧的标题文本。此方法允许您直接传递一个字符串资源 ID。
 *
 * @param titleRes 左侧标题文本的资源 ID。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setLeftTitle(@StringRes titleRes: Int): TitleBar =
    setLeftTitle(context.getString(titleRes))

/**
 * 设置标题栏左侧的格式化标题文本。
 *
 * 使用资源 ID 设置标题栏左侧的格式化标题文本。此方法允许您传递一个字符串资源 ID 和格式化参数。
 *
 * @param titleRes 左侧格式化标题文本的资源 ID。
 * @param formatArgs 格式化参数。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setLeftTitle(@StringRes titleRes: Int, vararg formatArgs: Any): TitleBar =
    setLeftTitle(context.getString(titleRes, *formatArgs))

/**
 * 设置标题栏左侧的标题文本。
 *
 * 使用 CharSequence 设置标题栏左侧的标题文本。此方法允许您直接传递 CharSequence 对象。
 *
 * @param title 要设置的左侧标题文本。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setLeftTitle(title: CharSequence?): TitleBar {
    mLeftView.text = title
    return this
}

/**
 * 获取标题栏左侧的标题文本。
 *
 * 此函数返回标题栏左侧设置的标题文本，如果左侧标题文本没有被设置，则可能返回 null。
 *
 * @return 返回标题栏左侧的标题文本，类型为 CharSequence?。
 */
fun TitleBar.getLeftTitle(): CharSequence? = mLeftView.text

/**
 * 设置标题栏右侧的标题文本。
 *
 * 使用资源 ID 设置标题栏右侧的标题文本。此方法允许您直接传递一个字符串资源 ID。
 *
 * @param titleRes 右侧标题文本的资源 ID。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setRightTitle(@StringRes titleRes: Int): TitleBar =
    setRightTitle(context.getString(titleRes))

/**
 * 设置标题栏右侧的格式化标题文本。
 *
 * 使用资源 ID 设置标题栏右侧的格式化标题文本。此方法允许您传递一个字符串资源 ID 和格式化参数。
 *
 * @param titleRes 右侧格式化标题文本的资源 ID。
 * @param formatArgs 格式化参数的可变参数。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setRightTitle(@StringRes titleRes: Int, vararg formatArgs: Any): TitleBar =
    setRightTitle(context.getString(titleRes, *formatArgs))

/**
 * 设置标题栏右侧的标题文本。
 *
 * 使用 CharSequence 设置标题栏右侧的标题文本。此方法允许您直接传递 CharSequence 对象。
 *
 * @param title 要设置的右侧标题文本。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setRightTitle(title: CharSequence?): TitleBar {
    mRightView.text = title
    return this
}

/**
 * 获取标题栏右侧的标题文本。
 *
 * @return 返回标题栏右侧的标题文本，如果没有设置则可能返回 null。
 */
fun TitleBar.getRightTitle(): CharSequence? = mRightView.text

/**
 * 设置标题栏标题的样式。
 *
 * 此函数接受一个样式整数参数，并将其转换为对应的 [Typeface] 对象和样式，然后应用到标题文本上。
 *
 * @param style 样式整数参数，用于设置标题文本的字体样式，如 [Typeface.BOLD]。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setTitleStyle(style: Int): TitleBar = setTitleStyle(style.asTypeface, style)

/**
 * 设置标题栏标题的样式。
 *
 * 此函数接受一个 [Typeface] 对象和样式整数参数，并将其应用到标题文本上。
 *
 * @param typeface 要应用的 Typeface 对象。
 * @param style 字体样式，如 [Typeface.BOLD] 或 [Typeface.ITALIC]。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setTitleStyle(typeface: Typeface, style: Int): TitleBar {
    mTitleView.setTypeface(typeface, style)
    return this
}

/**
 * 设置标题栏左侧标题的样式。
 *
 * 此函数接受一个样式整数参数，并将其转换为对应的 [Typeface] 对象和样式，然后应用到左侧标题文本上。
 *
 * @param style 样式整数参数，用于设置左侧标题文本的字体样式。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setLeftTitleStyle(style: Int): TitleBar = setLeftTitleStyle(style.asTypeface, style)

/**
 * 设置标题栏左侧标题的样式。
 *
 * 此函数接受一个 [Typeface] 对象和样式整数参数，并将其应用到左侧标题文本上。
 *
 * @param typeface 要应用的 Typeface 对象。
 * @param style 字体样式，如 [Typeface.BOLD] 或 [Typeface.ITALIC]。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setLeftTitleStyle(typeface: Typeface, style: Int): TitleBar {
    mLeftView.setTypeface(typeface, style)
    return this
}

/**
 * 设置标题栏右侧标题的样式。
 *
 * 此函数接受一个样式整数参数，并将其转换为对应的 [Typeface] 对象和样式，然后应用到右侧标题文本上。
 *
 * @param style 样式整数参数，用于设置右侧标题文本的字体样式。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setRightTitleStyle(style: Int): TitleBar = setRightTitleStyle(style.asTypeface, style)

/**
 * 设置标题栏右侧标题的样式。
 *
 * 此函数接受一个 [Typeface] 对象和样式整数参数，并将其应用到右侧标题文本上。
 *
 * @param typeface 要应用的 Typeface 对象。
 * @param style 字体样式，如 [Typeface.BOLD] 或 [Typeface.ITALIC]。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setRightTitleStyle(typeface: Typeface, style: Int): TitleBar {
    mRightView.setTypeface(typeface, style)
    return this
}

/**
 * 设置标题栏标题文本的溢出显示模式。
 *
 * 如果标题文本超出了显示区域，此函数设置文本的溢出显示方式。
 *
 * @param where 指定文本溢出时的处理方式，可能为 [TruncateAt.START]、[TruncateAt.MIDDLE]、[TruncateAt.END] 或 null。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setTitleOverflowMode(where: TruncateAt?): TitleBar {
    where?.let { mTitleView.applyEllipsize(where = where) }
    return this
}

/**
 * 设置标题栏左侧标题文本的溢出显示模式。
 *
 * 如果左侧标题文本超出了显示区域，此函数设置文本的溢出显示方式。
 *
 * @param where 指定文本溢出时的处理方式，可能为 [TruncateAt.START]、[TruncateAt.MIDDLE]、[TruncateAt.END] 或 null。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setLeftTitleOverflowMode(where: TruncateAt?): TitleBar {
    where?.let { mLeftView.applyEllipsize(where = where) }
    return this
}

/**
 * 设置标题栏右侧标题文本的溢出显示模式。
 *
 * 如果右侧标题文本超出了显示区域，此函数设置文本的溢出显示方式。
 *
 * @param where 指定文本溢出时的处理方式，可能为 [TruncateAt.START]、[TruncateAt.MIDDLE]、[TruncateAt.END] 或 null。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setRightTitleOverflowMode(where: TruncateAt?): TitleBar {
    where?.let { mRightView.applyEllipsize(where = where) }
    return this
}

/**
 * 设置标题栏标题文本的颜色。
 *
 * 此函数接受一个颜色整数，并将其包装在一个 [ColorStateList] 对象中，然后应用到标题文本上。
 *
 * @param titleColor 标题文本的颜色值。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setTitleColor(titleColor: Int): TitleBar =
    setTitleColor(ColorStateList.valueOf(titleColor))

/**
 * 设置标题栏标题文本的颜色。
 *
 * 此函数接受一个 [ColorStateList] 对象，并将其应用到标题文本上。
 *
 * @param color 标题文本的颜色状态列表。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setTitleColor(color: ColorStateList?): TitleBar {
    color?.let { mTitleView.setTextColor(it) }
    return this
}

/**
 * 设置标题栏左侧标题文本的颜色。
 *
 * 此函数接受一个颜色整数，并将其包装在一个 [ColorStateList] 对象中，然后应用到左侧标题文本上。
 *
 * @param leftTitleColor 左侧标题文本的颜色值。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setLeftTitleColor(leftTitleColor: Int): TitleBar =
    setLeftTitleColor(ColorStateList.valueOf(leftTitleColor))

/**
 * 设置标题栏左侧标题文本的颜色。
 *
 * 此函数接受一个 [ColorStateList] 对象，并将其应用到左侧标题文本上。
 *
 * @param color 左侧标题文本的颜色状态列表。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setLeftTitleColor(color: ColorStateList?): TitleBar {
    color?.let { mLeftView.setTextColor(it) }
    return this
}

/**
 * 设置标题栏右侧标题文本的颜色。
 *
 * 此函数接受一个颜色整数，并将其包装在一个 [ColorStateList] 对象中，然后应用到右侧标题文本上。
 *
 * @param rightTitleColor 右侧标题文本的颜色值。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setRightTitleColor(rightTitleColor: Int): TitleBar =
    setRightTitleColor(ColorStateList.valueOf(rightTitleColor))

/**
 * 设置标题栏右侧标题文本的颜色。
 *
 * 此函数接受一个 [ColorStateList] 对象，并将其应用到右侧标题文本上。
 *
 * @param color 右侧标题文本的颜色状态列表。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setRightTitleColor(color: ColorStateList?): TitleBar {
    color?.let { mRightView.setTextColor(it) }
    return this
}

/**
 * 设置标题栏标题文本的字号大小。
 *
 * 此函数接受一个单位 [unit] 和字号大小 [size]，然后应用到标题文本上。
 *
 * @param unit 字号大小的单位，默认为 [TypedValue.COMPLEX_UNIT_PX]。
 * @param size 要设置的字号大小。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setTitleSize(unit: Int = COMPLEX_UNIT_PX, size: Float): TitleBar {
    mTitleView.setTextSize(unit, size)
    return this
}

/**
 * 设置标题栏左侧标题文本的字号大小。
 *
 * 此函数接受一个单位 [unit] 和字号大小 [size]，然后应用到左侧标题文本上。
 *
 * @param unit 字号大小的单位，默认为 [TypedValue.COMPLEX_UNIT_PX]。
 * @param size 要设置的字号大小。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setLeftTitleSize(unit: Int = COMPLEX_UNIT_PX, size: Float): TitleBar {
    mLeftView.setTextSize(unit, size)
    return this
}

/**
 * 设置标题栏右侧标题文本的字号大小。
 *
 * 此函数接受一个单位 [unit] 和字号大小 [size]，然后应用到右侧标题文本上。
 *
 * @param unit 字号大小的单位，默认为 [TypedValue.COMPLEX_UNIT_PX]。
 * @param size 要设置的字号大小。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setRightTitleSize(unit: Int = COMPLEX_UNIT_PX, size: Float): TitleBar {
    mRightView.setTextSize(unit, size)
    return this
}

/**
 * 设置标题栏标题图标。
 *
 * 此函数接受一个图标资源 ID，并使用 [Context.getDrawableCompat] 获取 Drawable 对象，然后应用到标题栏的标题图标上。
 *
 * @param iconResId 标题图标的资源 ID。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setTitleIcon(@DrawableRes iconResId: Int): TitleBar =
    setTitleIcon(ResourcesCompat.getDrawable(context.resources, iconResId,context.theme))

/**
 * 设置标题栏标题图标。
 *
 * 此函数接受一个 Drawable 对象，并将其应用到标题栏的标题图标上，同时可以应用颜色和尺寸设置。
 *
 * @param drawable 标题图标的 Drawable 对象。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setTitleIcon(drawable: Drawable?): TitleBar {
    drawable?.applyTint(shouldApplyTint = true, tint = mTitleIconTint)
    drawable?.applySize(width = mTitleIconWidth, height = mTitleIconHeight)
    mTitleView.setCompoundDrawableGravity(drawable = drawable, gravity = mTitleIconGravity)
    return this
}

/**
 * 获取标题栏当前的标题图标。
 *
 * @return 返回当前设置的标题图标 Drawable 对象，如果没有设置则返回 null。
 */
fun TitleBar.getTitleIcon(): Drawable? =
    mTitleView.getCompoundDrawableGravity(gravity = mTitleIconGravity)

/**
 * 设置标题栏左侧图标。
 *
 * 此函数接受一个图标资源 ID，并使用 [Context.getDrawableCompat] 获取 Drawable 对象，然后应用到标题栏的左侧图标上。
 *
 * @param iconResId 左侧图标的资源 ID。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setLeftIcon(@DrawableRes iconResId: Int): TitleBar =
    setLeftIcon(ResourcesCompat.getDrawable(context.resources, iconResId,context.theme))

/**
 * 设置标题栏左侧图标。
 *
 * 此函数接受一个 Drawable 对象，并将其应用到标题栏的左侧图标上，同时可以应用颜色和尺寸设置。
 *
 * @param drawable 左侧图标的 Drawable 对象。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setLeftIcon(drawable: Drawable?): TitleBar {
    drawable?.applyTint(shouldApplyTint = true, tint = mLeftIconTint)
    drawable?.applySize(width = mLeftIconWidth, height = mLeftIconHeight)
    mLeftView.setCompoundDrawableGravity(drawable = drawable, gravity = mLeftIconGravity)
    return this
}

/**
 * 获取标题栏当前的左侧图标。
 *
 * @return 返回当前设置的左侧图标 Drawable 对象，如果没有设置则返回 null。
 */
fun TitleBar.getLeftIcon(): Drawable? =
    mLeftView.getCompoundDrawableGravity(gravity = mLeftIconGravity)

/**
 * 设置标题栏右侧图标。
 *
 * 此函数接受一个图标资源 ID，并使用 [Context.getDrawableCompat] 获取 Drawable 对象，然后应用到标题栏的右侧图标上。
 *
 * @param iconResId 右侧图标的资源 ID。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setRightIcon(@DrawableRes iconResId: Int): TitleBar =
    setRightIcon(ResourcesCompat.getDrawable(context.resources, iconResId,context.theme))

/**
 * 设置标题栏右侧图标。
 *
 * 此函数接受一个 Drawable 对象，并将其应用到标题栏的右侧图标上，同时可以应用颜色和尺寸设置。
 *
 * @param drawable 右侧图标的 Drawable 对象。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setRightIcon(drawable: Drawable?): TitleBar {
    drawable?.applyTint(shouldApplyTint = true, tint = mRightIconTint)
    drawable?.applySize(width = mRightIconWidth, height = mRightIconHeight)
    mRightView.setCompoundDrawableGravity(drawable = drawable, gravity = mRightIconGravity)
    return this
}

/**
 * 获取标题栏当前的右侧图标。
 *
 * @return 返回当前设置的右侧图标 Drawable 对象，如果没有设置则返回 null。
 */
fun TitleBar.getRightIcon(): Drawable? =
    mRightView.getCompoundDrawableGravity(gravity = mRightIconGravity)

/**
 * 设置标题栏标题图标的尺寸。
 *
 * 此函数设置标题图标的宽度和高度，并应用到当前的标题图标上。
 *
 * @param width 图标宽度。
 * @param height 图标高度。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setTitleIconSize(width: Int, height: Int): TitleBar {
    mTitleIconWidth = width
    mTitleIconHeight = height
    getTitleIcon()?.applySize(width, height)
    return this
}

/**
 * 设置标题栏左侧图标的尺寸。
 *
 * 此函数设置左侧图标的宽度和高度，并应用到当前的左侧图标上。
 *
 * @param width 图标宽度。
 * @param height 图标高度。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setLeftIconSize(width: Int, height: Int): TitleBar {
    mLeftIconWidth = width
    mLeftIconHeight = height
    getLeftIcon()?.applySize(width, height)
    return this
}

/**
 * 设置标题栏右侧图标的尺寸。
 *
 * 此函数设置右侧图标的宽度和高度，并应用到当前的右侧图标上。
 *
 * @param width 图标宽度。
 * @param height 图标高度。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setRightIconSize(width: Int, height: Int): TitleBar {
    mRightIconWidth = width
    mRightIconHeight = height
    getRightIcon()?.applySize(width, height)
    return this
}

/**
 * 设置标题栏标题图标与文本之间的间距。
 *
 * 此函数设置标题图标和其旁边文本的间距，影响标题栏标题部分的视觉效果。
 *
 * @param padding 图标与文本之间的间距大小。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setTitleIconPadding(padding: Int): TitleBar {
    mTitleView.compoundDrawablePadding = padding
    return this
}

/**
 * 设置标题栏左侧图标与文本之间的间距。
 *
 * 此函数设置左侧图标和其旁边文本的间距，影响标题栏左侧部分的视觉效果。
 *
 * @param padding 图标与文本之间的间距大小。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setLeftIconPadding(padding: Int): TitleBar {
    mLeftView.compoundDrawablePadding = padding
    return this
}

/**
 * 设置标题栏右侧图标与文本之间的间距。
 *
 * 此函数设置右侧图标和其旁边文本的间距，影响标题栏右侧部分的视觉效果。
 *
 * @param padding 图标与文本之间的间距大小。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setRightIconPadding(padding: Int): TitleBar {
    mRightView.compoundDrawablePadding = padding
    return this
}

/**
 * 设置标题栏标题图标的颜色。
 *
 * 此函数设置标题图标的颜色，并更新当前的标题图标以应用新颜色。
 *
 * @param color 图标颜色值。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setTitleIconTint(color: Int): TitleBar {
    mTitleIconTint = color
    getTitleIcon()?.applyTint(shouldApplyTint = true, tint = color)
    return this
}

/**
 * 清除标题栏标题图标的颜色。
 *
 * 此函数清除之前设置的标题图标颜色，恢复为默认状态。
 *
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.clearTitleIconTint(): TitleBar {
    mTitleIconTint = Color.TRANSPARENT
    getTitleIcon()?.applyTint(shouldApplyTint = false)
    return this
}

/**
 * 设置标题栏左侧图标的颜色。
 *
 * 此函数设置左侧图标的颜色，并更新当前的左侧图标以应用新颜色。
 *
 * @param color 图标颜色值。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setLeftIconTint(color: Int): TitleBar {
    mLeftIconTint = color
    getLeftIcon()?.applyTint(shouldApplyTint = true, tint = color)
    return this
}

/**
 * 清除标题栏左侧图标的颜色。
 *
 * 此函数清除之前设置的左侧图标颜色，恢复为默认状态。
 *
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.clearLeftIconTint(): TitleBar {
    mLeftIconTint = Color.TRANSPARENT
    getLeftIcon()?.applyTint(shouldApplyTint = false)
    return this
}

/**
 * 设置标题栏右侧图标的颜色。
 *
 * 此函数设置右侧图标的颜色，并更新当前的右侧图标以应用新颜色。
 *
 * @param color 图标颜色值。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setRightIconTint(color: Int): TitleBar {
    mRightIconTint = color
    getRightIcon()?.applyTint(shouldApplyTint = true, tint = color)
    return this
}

/**
 * 清除标题栏右侧图标的颜色。
 *
 * 此函数清除之前设置的右侧图标颜色，恢复为默认状态。
 *
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.clearRightIconTint(): TitleBar {
    mRightIconTint = Color.TRANSPARENT
    getRightIcon()?.applyTint(shouldApplyTint = false)
    return this
}

/**
 * 设置标题栏标题图标的重力对齐方式。
 *
 * 此函数设置标题图标相对于文本的对齐方式，例如居中、左侧或右侧等。
 *
 * @param gravity 对齐方式，如 [Gravity.CENTER]、[Gravity.LEFT] 或 [Gravity.RIGHT]。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setTitleIconGravity(gravity: Int): TitleBar {
    val drawable = getTitleIcon()
    mTitleIconGravity = gravity
    drawable?.let { mTitleView.setCompoundDrawableGravity(gravity = gravity, drawable = it) }
    return this
}

/**
 * 设置标题栏左侧图标的重力对齐方式。
 *
 * 此函数设置左侧图标相对于文本的对齐方式。
 *
 * @param gravity 对齐方式，如 [Gravity.CENTER]、[Gravity.LEFT] 或 [Gravity.RIGHT]。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setLeftIconGravity(gravity: Int): TitleBar {
    val drawable = getLeftIcon()
    mLeftIconGravity = gravity
    drawable?.let { mLeftView.setCompoundDrawableGravity(gravity = gravity, drawable = it) }
    return this
}

/**
 * 设置标题栏右侧图标的重力对齐方式。
 *
 * 此函数设置右侧图标相对于文本的对齐方式。
 *
 * @param gravity 对齐方式，如 [Gravity.CENTER]、[Gravity.LEFT] 或 [Gravity.RIGHT]。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setRightIconGravity(gravity: Int): TitleBar {
    val drawable = getRightIcon()
    mRightIconGravity = gravity
    drawable?.let { mRightView.setCompoundDrawableGravity(gravity = gravity, drawable = it) }
    return this
}

/**
 * 设置标题栏左侧元素的背景。
 *
 * 此函数通过资源 ID 设置左侧元素的背景，并使用 [Context.getDrawableCompat] 获取 Drawable 对象。
 *
 * @param backgroundResId 左侧元素背景的资源 ID。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setLeftBackground(@DrawableRes backgroundResId: Int): TitleBar =
    setLeftBackground(ResourcesCompat.getDrawable(context.resources, backgroundResId,context.theme))

/**
 * 设置标题栏左侧元素的背景。
 *
 * 此函数接受一个 Drawable 对象，并将其设置为左侧元素的背景。
 *
 * @param background 左侧元素的背景 Drawable 对象。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setLeftBackground(background: Drawable?): TitleBar {
    mLeftView.background = background
    return this
}

/**
 * 设置标题栏右侧元素的背景。
 *
 * 此函数通过资源 ID 设置右侧元素的背景，并使用 [Context.getDrawableCompat] 获取 Drawable 对象。
 *
 * @param backgroundResId 右侧元素背景的资源 ID。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setRightBackground(@DrawableRes backgroundResId: Int): TitleBar =
    setRightBackground(ResourcesCompat.getDrawable(context.resources,backgroundResId,context.theme))

/**
 * 设置标题栏右侧元素的背景。
 *
 * 此函数接受一个 Drawable 对象，并将其设置为右侧元素的背景。
 *
 * @param background 右侧元素的背景 Drawable 对象。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setRightBackground(background: Drawable?): TitleBar {
    mRightView.background = background
    return this
}

/**
 * 设置标题栏左侧元素的前景。
 *
 * 此函数通过资源 ID 设置左侧元素的前景，并使用 [Context.getDrawableCompat] 获取 Drawable 对象。
 * 仅在 Android 6.0（API 级别 23）及以上版本上支持前景设置。
 *
 * @param foregroundResId 左侧元素前景的资源 ID。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setLeftForeground(@DrawableRes foregroundResId: Int): TitleBar =
    setLeftForeground(ResourcesCompat.getDrawable(context.resources, foregroundResId,context.theme))

/**
 * 设置标题栏左侧元素的前景。
 *
 * 此函数接受一个 Drawable 对象，并将其设置为左侧元素的前景。
 * 仅在 Android 6.0（API 级别 23）及以上版本上支持前景设置。
 *
 * @param foreground 左侧元素的前景 Drawable 对象。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setLeftForeground(foreground: Drawable?): TitleBar {
    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) mLeftView.foreground = foreground
    return this
}

/**
 * 设置标题栏右侧元素的前景。
 *
 * 此函数通过资源 ID 设置标题栏右侧元素的前景 Drawable，并使用 [ContextCompat.getDrawableCompat] 获取 Drawable 对象。
 * 仅在 Android 6.0（API 级别 23）及以上版本上支持前景设置。
 *
 * @param foregroundResId 右侧元素前景的资源 ID。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setRightForeground(@DrawableRes foregroundResId: Int): TitleBar =
    setRightForeground(ResourcesCompat.getDrawable(context.resources, foregroundResId,context.theme))

/**
 * 设置标题栏右侧元素的前景。
 *
 * 此函数接受一个 Drawable 对象，并将其设置为标题栏右侧元素的前景。
 * 仅在 Android 6.0（API 级别 23）及以上版本上支持前景设置。
 *
 * @param foreground 右侧元素的前景 Drawable 对象。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setRightForeground(foreground: Drawable?): TitleBar {
    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) mRightView.foreground = foreground
    return this
}

/**
 * 设置标题栏底部线条的可见性。
 *
 * @param visible 如果为 true，则显示线条；如果为 false，则隐藏线条。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setLineVisible(visible: Boolean): TitleBar {
    mLineView.setVisible(visible)
    return this
}

/**
 * 设置标题栏底部线条的颜色。
 *
 * @param color 线条的颜色值。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setLineColor(color: Int): TitleBar = setLineDrawable(ColorDrawable(color))

/**
 * 设置标题栏底部线条的 Drawable。
 *
 * 此函数接受一个 Drawable 对象，并将其设置为标题栏底部线条的背景。
 *
 * @param drawable 线条的 Drawable 对象。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setLineDrawable(drawable: Drawable?): TitleBar {
    drawable?.let { mLineView.background = it }
    return this
}

/**
 * 设置标题栏底部线条的大小。
 *
 * 此函数设置标题栏底部线条的高度。
 *
 * @param px 线条的高度，以像素为单位。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setLineSize(px: Int): TitleBar {
    val layoutParams = mLineView.layoutParams
    layoutParams.height = px
    mLineView.layoutParams = layoutParams
    return this
}

/**
 * 设置标题栏标题的重力对齐方式。
 *
 * 此函数根据给定的 [gravity] 参数设置标题文本在其容器内的对齐方式。同时，根据当前布局方向，检查并确保标题居中时左右两侧不包含内容。
 *
 * @param gravity 标题的对齐方式，如 [Gravity.LEFT]、[Gravity.CENTER] 或 [Gravity.RIGHT]。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setTitleGravity(gravity: Int): TitleBar {
    val absoluteGravity = getAbsoluteGravity(gravity = gravity)
    if (absoluteGravity == Gravity.LEFT && (if (context.isLayoutRtl) mRightView else mLeftView).containsContent) {
        e("Title center of gravity for the left, the left title can not have content")
        return this
    }
    if (absoluteGravity == Gravity.RIGHT && (if (context.isLayoutRtl) mLeftView else mRightView).containsContent) {
        e("Title center of gravity for the right, the right title can not have content")
        return this
    }
    val layoutParams = mTitleView.layoutParams as FrameLayout.LayoutParams
    layoutParams.gravity = absoluteGravity
    mTitleView.layoutParams = layoutParams
    return this
}

/**
 * 设置标题栏各个子视图的水平内边距。
 *
 * 此函数分别设置标题栏左侧、标题和右侧子视图的水平内边距。同时，垂直内边距保持不变。
 *
 * @param leftHorizontalPadding 左侧子视图的水平内边距。
 * @param titleHorizontalPadding 标题子视图的水平内边距。
 * @param rightHorizontalPadding 右侧子视图的水平内边距。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setChildHorizontalPadding(
    leftHorizontalPadding: Int,
    titleHorizontalPadding: Int,
    rightHorizontalPadding: Int
): TitleBar {
    mLeftHorizontalPadding = leftHorizontalPadding
    mTitleHorizontalPadding = titleHorizontalPadding
    mRightHorizontalPadding = rightHorizontalPadding
    mLeftView.setPadding(
        mLeftHorizontalPadding,
        mVerticalPadding,
        mLeftHorizontalPadding,
        mVerticalPadding
    )
    mTitleView.setPadding(
        mTitleHorizontalPadding,
        mVerticalPadding,
        mTitleHorizontalPadding,
        mVerticalPadding
    )
    mRightView.setPadding(
        mRightHorizontalPadding,
        mVerticalPadding,
        mRightHorizontalPadding,
        mVerticalPadding
    )
    return this
}

/**
 * 设置标题栏各个子视图的垂直内边距。
 *
 * 此函数统一设置标题栏左侧、标题和右侧子视图的垂直内边距为相同的值。水平内边距保持不变。
 *
 * @param verticalPadding 要设置的垂直内边距值。
 * @return 返回 TitleBar 实例自身，允许链式调用。
 */
fun TitleBar.setChildVerticalPadding(verticalPadding: Int): TitleBar {
    mVerticalPadding = verticalPadding
    mLeftView.setPadding(
        mLeftHorizontalPadding,
        verticalPadding,
        mLeftHorizontalPadding,
        verticalPadding
    )
    mTitleView.setPadding(
        mTitleHorizontalPadding,
        verticalPadding,
        mTitleHorizontalPadding,
        verticalPadding
    )
    mRightView.setPadding(
        mRightHorizontalPadding,
        verticalPadding,
        mRightHorizontalPadding,
        verticalPadding
    )
    return this
}

val TitleBar.leftTitleView
    get() = mLeftView

val TitleBar.titleView
    get() = mTitleView

val TitleBar.rightTitleView
    get() = mRightView

val TitleBar.lineView
    get() = mLineView

