package com.liecoder.framework.ktx

import android.text.InputFilter
import android.text.Spanned
import android.widget.EditText


/**
 * Adds a filter to an EditText to prevent the entry of emoji characters.
 *
 * This function utilizes the [addFilter] method to add a character filter that checks each character
 * to determine if it is an emoji. If a character is an emoji, it will be blocked from entry.
 *
 * @return The EditText instance itself, allowing for chaining of method calls.
 */
fun EditText.filterEmoji(): EditText {
    return addFilter { it.isEmoji() }
}

/**
 * 过滤掉 EditText 中输入的特定字符集。
 *
 * 此函数接受一组字符，并将它们添加到过滤条件中，阻止这些字符被输入到 EditText 中。
 *
 * @param characters 一个字符序列，包含需要被过滤掉的字符。
 * @return 返回 EditText 实例自身，允许链式调用。
 */
fun EditText.filterCharacters(vararg characters: Char): EditText {
    return addFilter(characters.toSet()::contains)
}

/**
 * 为 EditText 添加一个基于单个字符的过滤条件。
 *
 * 此函数接受一个字符筛选谓词 [filterPredicate]，该谓词返回 true 以允许字符输入，返回 false 以阻止字符输入。
 *
 * @param filterPredicate 一个接收单个字符并返回 Boolean 值的函数，用于决定是否允许该字符输入。
 * @return 返回 EditText 实例自身，允许链式调用。
 */
fun EditText.addFilter(filterPredicate: (Char) -> Boolean): EditText {
    filters += InputFilter { source, start, end, _, _, _ ->
        if (source is Spanned) {
            // 如果输入是 Spanned 类型，直接返回它，不对富文本内容进行过滤
            return@InputFilter source
        }
        // 遍历输入的字符序列
        for (i in start until end) {
            // 如果字符应该被过滤，则返回空字符串
            if (filterPredicate(source[i])) {
                return@InputFilter ""
            }
        }
        // 如果没有字符被过滤，则允许输入
        null
    }
    return this
}