package com.liecoder.framework.ktx

import com.google.gson.Gson
import com.google.gson.GsonBuilder
import com.google.gson.reflect.TypeToken


/**
 * 创建一个 `Gson` 实例，根据 [includeNulls] 参数决定是否序列化 `null` 值。
 *
 * @param includeNulls 当设置为 `true` 时，`Gson` 实例将序列化 `null` 值；否则，`null` 值在序列化时会被忽略。
 * @return 返回配置好的 `Gson` 实例。
 */
fun gson(includeNulls: Boolean): Gson {
    return if (includeNulls) {
        GsonBuilder().create()
    } else {
        GsonBuilder().serializeNulls().create()
    }
}

/**
 * 将任意对象转换为 JSON 字符串。
 *
 * @param includeNulls 是否在 JSON 字符串中包含 `null` 值，默认为 `true`。
 * @return 返回表示该对象的 JSON 字符串。
 */
fun Any.toJson(includeNulls: Boolean = true): String {
    return gson(includeNulls).toJson(this)
}

/**
 * 将 JSON 字符串转换为指定类型的 Java 对象。
 *
 * 使用 Gson 库将字符串反序列化到一个对象。此函数使用泛型 [T] 来指定目标类型。
 *
 * @param T 目标 Java 对象的类型。
 * @param includeNulls 是否在反序列化时处理 `null` 值，默认为 `true`。
 * @return 返回反序列化后的 Java 对象。
 */
inline fun <reified T> String.toBean(includeNulls: Boolean = true): T {
    return gson(includeNulls).fromJson(this, object : TypeToken<T>() {}.type)
}


/**
 * 尝试将 JSON 字符串转换为指定类型的 Java 对象，如果转换失败则返回 `null`。
 *
 * 使用 `Gson` 库将字符串反序列化为一个对象。与 [toBean] 函数不同，如果反序列化过程中发生异常，此函数不会抛出异常，而是返回 `null`。
 * 这在处理可能损坏或不完整的 JSON 数据时非常有用。
 *
 * @param T 目标 Java 对象的类型。
 * @param includeNulls 是否在反序列化时处理 `null` 值，默认为 `true`。
 * @return 返回反序列化后的 Java 对象，或者在发生异常时返回 `null`。
 */
inline fun <reified T> String.toBeanOrNull(includeNulls: Boolean = true): T? =
    runCatching { toBean<T>(includeNulls) }.onFailure { it.printStackTrace() }.getOrNull()