package com.liecoder.framework.ktx

import java.util.regex.Pattern

/**
 * 验证给定的 CharSequence 是否是有效的电子邮件地址。
 * 该函数首先检查 CharSequence 是否为空或包含中文字符，然后使用正则表达式验证电子邮件格式。
 *
 * @param charSequence 待验证的字符串。
 * @return 如果字符串是有效的电子邮件地址，则返回 true；否则返回 false。
 *
 * 示例：
 * ```
 * "example@email.com".isValidEmail() // 返回 true
 * "例子@邮件.com".isValidEmail() // 返回 false，因为包含中文字符
 * "".isValidEmail() // 返回 false，因为是一个空字符串
 * ```
 */
fun CharSequence.isValidEmail(): Boolean = takeIf { !hasAnyChinese() && !isNullOrEmpty() }?.let {
    Pattern
        .compile("\\w+([-+.]\\w+)*@\\w+([-.]\\w+)*\\.\\w+([-.]\\w+)*")
        .matcher(it)
        .matches()
} ?: false

/**
 * 验证给定的 CharSequence 是否为有效的中国手机号码。
 * 手机号码必须是11位数字，以1开头，第二位是3到9之间的数字，后面跟着其他9位数字。
 * 可选的国际区号为 +86 或 00。
 *
 * @param charSequence 待验证的字符串。
 * @return 如果字符串是有效的手机号码，则返回 true；否则返回 false。
 *
 * 示例：
 * ```
 * "13812345678".isValidPhoneNumber() // 返回 true
 * "+8613812345678".isValidPhoneNumber() // 返回 true
 * "00861381234567".isValidPhoneNumber() // 返回 true
 * ```
 */
fun CharSequence.isValidPhoneNumber(): Boolean = Pattern
    .compile("^(?:(?:\\+|00)86)?1[3-9]\\d{9}$")
    .matcher(this)
    .matches()

/**
 * 验证给定的 CharSequence 是否是有效的中国统一社会信用代码。
 * 中国统一社会信用代码通常由15位、18位或20位的数字和大写字母组成。
 *
 * @return 如果 CharSequence 符合社会信用代码的格式，则返回 true；否则返回 false。
 *
 * 示例：
 * ```
 * "91310000MA1JKBY9K".isValidSocialCreditCode() // 返回 true 或 false
 * ```
 */
fun CharSequence.isValidCreditCode(): Boolean = Pattern
    .compile("^[0-9A-Z]{15}|^[0-9A-Z]{18}|^[0-9A-Z]{20}$")
    .matcher(this)
    .matches()


/**
 * 检查 CharSequence 中是否包含任意中文字符。
 * @return 如果 CharSequence 包含至少一个中文字符，则返回 true；否则返回 false。
 */
fun CharSequence.hasAnyChinese(): Boolean = Pattern
    .compile("[\u4e00-\u9fa5]") //匹配常见的中文字符范围 [\u4e00-\u9fa5]
    .matcher(this)
    .find()

/**
 * 过滤掉 CharSequence 中的所有表情符号。
 * @return 返回一个新的 CharSequence 对象，其中不包含任何表情符号。
 */
fun CharSequence.filterEmoji(): CharSequence = filterNot { it.isEmoji() }

/**
 * 确定当前字符串（作为子字符串）在给定的父字符串中的位置范围。
 * 如果当前字符串不在父字符串中，或者子字符串的结束位置超出父字符串的边界，则返回 null。
 *
 * @param parent 父字符串，当前字符串需要在其中搜索。
 * @return 返回一个表示当前字符串在父字符串中位置的 IntRange 对象，或者如果未找到或超出边界，则返回 null。
 */
fun String.rangeWithin(parent: String): IntRange? {
    val start = parent.indexOf(this)
    if (start == -1) return null
    val end = start + length - 1
    if (end > parent.length) return null
    return start..end
}


/**
 * 对手机号码进行加密处理，以保护用户隐私。
 * 假设手机号码长度为11位，将中间的4位数字替换为星号（****）。
 * 如果传入的字符串不是手机号码长度，则直接返回原字符串。
 *
 * @return 加密后的手机号码或原始字符串。
 */
fun String.encryptPhoneNumber(): String =
    if (length == 11) substring(0, 3) + "****" + substring(7) else this


/**
 * 生成一个指定长度的随机字符串。
 * @param length 随机字符串的长度。
 * @return 返回一个由大写字母、小写字母和数字组成的随机字符串。
 */
fun random(length: Int): String {
    val allowedChars = ('A'..'Z') + ('a'..'z') + ('0'..'9')
    return (1..length).map { allowedChars.random() }.joinToString("")
}
