package com.liecoder.framework.titlebar

import android.graphics.drawable.Drawable
import android.graphics.drawable.StateListDrawable


/**
 * Author: lieco(liecoder@163.com)
 * Date: 2024/8/18
 * Version: 1.0
 */
class SelectorDrawable : StateListDrawable() {

    class Builder {

        // 定义各种状态下的 Drawable 对象
        private var mDefault: Drawable? = null
        private var mFocused: Drawable? = null
        private var mPressed: Drawable? = null
        private var mChecked: Drawable? = null
        private var mEnabled: Drawable? = null
        private var mSelected: Drawable? = null
        private var mHovered: Drawable? = null

        /**
         * 设置默认状态下的 Drawable。
         *
         * @param default Drawable 对象。
         * @return Builder 实例自身，允许链式调用。
         */
        fun setDefault(default: Drawable): Builder {
            mDefault = default
            return this
        }

        /**
         * 设置获得焦点状态下的 Drawable。
         *
         * @param focused Drawable 对象。
         * @return Builder 实例自身，允许链式调用。
         */
        fun setFocused(focused: Drawable): Builder {
            mFocused = focused
            return this
        }

        /**
         * 设置点击状态下的 Drawable。
         *
         * @param pressed Drawable 对象。
         * @return Builder 实例自身，允许链式调用。
         */
        fun setPressed(pressed: Drawable): Builder {
            mPressed = pressed
            return this
        }

        /**
         * 设置选中状态下的 Drawable。
         *
         * @param checked Drawable 对象。
         * @return Builder 实例自身，允许链式调用。
         */
        fun setChecked(checked: Drawable): Builder {
            mChecked = checked
            return this
        }

        /**
         * 设置启用状态下的 Drawable。
         *
         * @param enabled Drawable 对象。
         * @return Builder 实例自身，允许链式调用。
         */
        fun setEnabled(enabled: Drawable): Builder {
            mEnabled = enabled
            return this
        }

        /**
         * 设置选中状态下的 Drawable。
         *
         * @param selected Drawable 对象。
         * @return Builder 实例自身，允许链式调用。
         */
        fun setSelected(selected: Drawable): Builder {
            mSelected = selected
            return this
        }

        fun setHovered(hovered: Drawable): Builder {
            mHovered = hovered
            return this
        }

        /**
         * 构建并返回一个配置好的 [SelectorDrawable] 对象。
         *
         * @return 配置好的 [SelectorDrawable] 对象。
         */
        fun build(): SelectorDrawable {
            val selectorDrawable = SelectorDrawable()
            mFocused?.run {
                selectorDrawable.addState(intArrayOf(android.R.attr.state_focused), this)
            }
            mPressed?.run {
                selectorDrawable.addState(intArrayOf(android.R.attr.state_pressed), this)
            }
            mChecked?.run {
                selectorDrawable.addState(intArrayOf(android.R.attr.state_checked), this)
            }
            mEnabled?.run {
                selectorDrawable.addState(intArrayOf(android.R.attr.state_enabled), this)
            }
            mSelected?.run {
                selectorDrawable.addState(intArrayOf(android.R.attr.state_selected), this)
            }
            mHovered?.run {
                selectorDrawable.addState(intArrayOf(android.R.attr.state_hovered), this)
            }
            mDefault?.run { selectorDrawable.addState(intArrayOf(), this) }
            return selectorDrawable
        }
    }

    companion object {
        /**
         * 创建 SelectorDrawable 的构建器实例。
         *
         * @return SelectorDrawable 的 Builder 实例。
         */
        fun builder(): Builder {
            return Builder()
        }
    }

}