package com.liecoder.framework.base

import android.view.Gravity
import android.view.View
import android.view.animation.Animation
import razerdp.util.animation.AnimationHelper
import razerdp.util.animation.TranslationConfig

/**
 * 基础 PopupWindow服务接口，定义了管理和控制PopupWindow视图显示的通用方法。
 *
 * 此接口提供了一套标准方法，用于初始化视图、更新视图内容、显示和隐藏PopupWindow，以及定义PopupWindow的动画效果。
 * 实现此接口的类应负责PopupWindow视图的生命周期和行为。
 *
 * @author liecoder
 * @date 2024/7/3 周三
 * @version 1.0
 */
interface IBaseWindowService {

    /**
     * 初始化PopupWindow视图的方法。
     *
     * <p>在这个方法中，应执行所有必要的步骤来初始化PopupWindow视图，例如设置布局、绑定数据和配置视图组件。
     */
    fun initView()


    /**
     * 更新PopupWindow视图的方法。
     *
     * <p>此方法用于在数据变化时刷新PopupWindow视图，确保视图显示的信息是最新的。
     */
    fun updateView()


    /**
     * 显示PopupWindow的方法。
     *
     * <p>此方法用于将PopupWindow显示在屏幕上。如果提供了 anchorView 参数，PopupWindow将根据该视图定位；如果没有提供，则使用默认位置。
     *
     * @param anchorView PopupWindow附着的视图，用于定位PopupWindow。如果为 null，则使用默认位置。
     */
    fun show(anchorView: View? = null)



    /**
     * 获取PopupWindow显示时的重力方向。
     *
     * <p>此方法返回一个整数，表示PopupWindow显示时的重力方向，例如 [Gravity.BOTTOM] 表示PopupWindow将显示在屏幕底部。
     * 默认实现返回 [Gravity.BOTTOM]，但可以根据需要重写以提供不同的重力方向。
     *
     * @return PopupWindow的重力方向。
     */
    fun gravity(): Int = Gravity.BOTTOM


    /**
     * 获取PopupWindow显示时的动画效果。
     *
     * <p>此方法返回一个 [Animation] 对象，定义了PopupWindow显示时的动画效果。默认实现提供从底部向上滑动的动画，
     * 但可以根据需要重写以提供不同的动画效果。
     *
     * @return PopupWindow显示时的动画效果。
     */
    fun showAnimation(): Animation? =
        AnimationHelper.asAnimation().withTranslation(TranslationConfig.FROM_BOTTOM).toShow()


    /**
     * 获取PopupWindow消失时的动画效果。
     *
     * <p>此方法返回一个 [Animation] 对象，定义了PopupWindow消失时的动画效果。默认实现提供从底部向下滑动的动画，
     * 但可以根据需要重写以提供不同的动画效果。
     *
     * @return PopupWindow消失时的动画效果。
     */
    fun dismissAnimation(): Animation? =
        AnimationHelper.asAnimation().withTranslation(TranslationConfig.TO_BOTTOM).toDismiss()

}