package com.liecoder.framework.ktx

import android.graphics.Canvas
import android.graphics.Color
import android.graphics.Paint
import android.graphics.Rect
import android.view.View
import androidx.recyclerview.widget.GridLayoutManager
import androidx.recyclerview.widget.LinearLayoutManager
import androidx.recyclerview.widget.RecyclerView
import androidx.recyclerview.widget.RecyclerView.Adapter
import androidx.recyclerview.widget.RecyclerView.ItemDecoration
import androidx.recyclerview.widget.RecyclerView.VERTICAL


/**
 * 设置 RecyclerView 使用垂直方向的线性布局管理器。
 *
 * 此扩展函数为 RecyclerView 设置一个 [LinearLayoutManager]，使得项目按照垂直方向排列。
 * 可以通过参数自定义是否反转布局，即从底部开始排列。
 *
 * @param reverseLayout 是否反转布局，即项目是从底部开始排列，默认为 `false`。
 * @return 返回 RecyclerView 实例自身，允许链式调用。
 */
fun RecyclerView.vertical(reverseLayout: Boolean = false): RecyclerView {
    layoutManager = LinearLayoutManager(context, VERTICAL, reverseLayout)
    return this
}

/**
 * 设置 RecyclerView 使用网格布局管理器。
 *
 * 此扩展函数为 RecyclerView 设置一个 [GridLayoutManager]，允许项目以网格的形式显示。
 * 可以通过参数自定义网格的列数、布局的方向、以及是否反转布局。
 *
 * @param spanCount 网格中每行的列数。
 * @param orientation 布局的方向，默认为垂直方向 [VERTICAL]。
 * @param reverseLayout 是否反转布局，即从末尾开始布局，默认为 `false`。
 * @return 返回 RecyclerView 实例自身，允许链式调用。
 */
fun RecyclerView.grid(
    spanCount: Int,
    @RecyclerView.Orientation orientation: Int = VERTICAL,
    reverseLayout: Boolean = false,
): RecyclerView {
    layoutManager = GridLayoutManager(context, spanCount, orientation, reverseLayout)
    return this
}

/**
 * 设置 RecyclerView 的适配器。
 *
 * 此扩展函数用于为 RecyclerView 设置一个适配器，适配器负责管理数据集合，并提供数据给 RecyclerView 显示。
 * 通过此函数设置适配器后，可以直接链式调用其他 RecyclerView 的设置方法。
 *
 * @param adapter 要设置的 Adapter 对象，必须继承自 Adapter 类。
 * @return 返回 RecyclerView 实例自身，允许链式调用。
 */
fun <T : Adapter<*>> RecyclerView.adapter(adapter: T): RecyclerView {
    this.adapter = adapter
    return this
}

/**
 * 为 RecyclerView 添加一个 ItemDecoration。
 *
 * 此扩展函数允许向 RecyclerView 添加一个 ItemDecoration 对象，用于自定义项的布局边界和绘制。
 * 与 [singleItemDecoration] 不同，此函数不会移除已存在的 ItemDecorations，而是直接添加新的装饰。
 *
 * @param itemDecoration 要添加的 ItemDecoration 对象，可以是自定义的装饰。
 * @return 返回 RecyclerView 实例自身，允许链式调用。
 */
fun <T : ItemDecoration> RecyclerView.itemDecoration(itemDecoration: T): RecyclerView {
    addItemDecoration(itemDecoration)
    return this
}

/**
 * 为 RecyclerView 设置单个 ItemDecoration。
 *
 * 此扩展函数允许为 RecyclerView 设置单个 ItemDecoration 对象，并确保之前所有的装饰被移除，只保留当前设置的装饰。
 *
 * @param itemDecoration 要设置的 ItemDecoration 对象，可以是自定义的装饰。
 * @return 返回 RecyclerView 实例自身，允许链式调用。
 */
fun <T : ItemDecoration> RecyclerView.singleItemDecoration(itemDecoration: T): RecyclerView {
    if (itemDecorationCount > 0) {
        for (i in 0 until itemDecorationCount) {
            removeItemDecorationAt(i)
        }
    }
    addItemDecoration(itemDecoration)
    return this
}

/**
 * 为 RecyclerView 添加边距装饰器（ItemDecoration）。
 *
 * 此扩展函数为 RecyclerView 添加边距装饰，可以设置垂直方向和水平方向的边距。
 * 垂直边距将应用于所有项，除非是第一项且 [includeEdge] 为 `false`。
 * 水平边距将应用于所有项的左边或右边。
 *
 * @param includeEdge 是否在列表的第一项和最后一项上也应用顶部和底部边距，默认为 `true`。
 * @param marginVertical 垂直边距的尺寸，以像素为单位，默认为 15dp 转换为像素。
 * @param marginStart 左边距的尺寸，以像素为单位，默认为 0。
 * @param marginEnd 右边距的尺寸，以像素为单位，默认为 0。
 * @return 返回 RecyclerView 实例自身，允许链式调用。
 */
fun RecyclerView.marginDecorate(
    includeEdge: Boolean = true,
    marginVertical: Int = 15.dpInt,
    marginStart: Int = 0,
    marginEnd: Int = 0
): RecyclerView {
    if (itemDecorationCount > 0) {
        for (i in 0 until itemDecorationCount) {
            removeItemDecorationAt(i)
        }
    }
    val itemCount = adapter?.itemCount ?: 0
    addItemDecoration(object : ItemDecoration() {
        override fun getItemOffsets(
            outRect: Rect,
            view: View,
            parent: RecyclerView,
            state: RecyclerView.State
        ) {
            if (parent.getChildLayoutPosition(view) < itemCount - 1) {
                outRect.bottom = marginVertical
            }
            if (parent.getChildLayoutPosition(view) == 0) {
                if (includeEdge) outRect.top = marginVertical
            } else {
                outRect.top = marginVertical
            }
            if (marginStart > 0) outRect.left = marginStart
            if (marginEnd > 0) outRect.right = marginEnd
        }
    })
    return this
}

/**
 * 为 RecyclerView 添加水平线条装饰器（ItemDecoration）。
 *
 * 此扩展函数为 RecyclerView 添加水平线条装饰，用于在列表项之间绘制分隔线。
 *
 * @param lineColor 分割线的颜色，默认为浅灰色（#E5E5E5）。
 * @param lineWidth 分割线的宽度，默认为0.5dp转换为像素。
 * @param marginStart 分割线到屏幕左边距的偏移，默认为54dp转换为像素。
 * @param marginEnd 分割线到屏幕右边距的偏移，默认为0dp。
 * @return 返回 RecyclerView 实例自身，允许链式调用。
 */
fun RecyclerView.lineDecorate(
    lineColor: Int = Color.parseColor("#E5E5E5"),
    lineWidth: Float = 0.5f.dpFloat,
    marginStart: Float = 54.dpFloat,
    marginEnd: Float = 0f
): RecyclerView {
    addItemDecoration(object : ItemDecoration() {
        override fun onDrawOver(
            c: Canvas,
            parent: RecyclerView,
            state: RecyclerView.State
        ) {
            val paint = Paint()
            paint.color = lineColor
            paint.strokeWidth = lineWidth
            paint.isAntiAlias = true
            val childCount = parent.childCount
            for (i in 0 until childCount - 1) {
                val child = parent.getChildAt(i)
                val startY = child.bottom.toFloat()
                val endX = child.right.toFloat() - marginEnd
                val endY = child.bottom.toFloat()
                c.drawLine(marginStart, startY, endX, endY, paint)
            }
        }
    })
    return this
}

/**
 * 为 RecyclerView 设置网格布局的装饰器（ItemDecoration）。
 *
 * 此函数为 RecyclerView 添加一个装饰器，使得其中的项目能够按照网格布局进行排列，并支持设置间距和是否包含边缘间距。
 *
 * @param spanCount 网格中每行的列数。
 * @param spacing 项目之间的间距，以像素为单位。
 * @param includeEdge 如果为 true，则第一列和最后一列的间距会与其它列不同，以实现边缘对齐的效果。
 * @return 返回 RecyclerView 实例自身，允许链式调用。
 */
fun RecyclerView.gridDecorate(
    spanCount: Int,
    spacing: Int,
    includeEdge: Boolean
): RecyclerView {
    addItemDecoration(object : RecyclerView.ItemDecoration() {
        override fun getItemOffsets(
            outRect: Rect,
            view: View,
            parent: RecyclerView,
            state: RecyclerView.State
        ) {
            val position = parent.getChildAdapterPosition(view) // item position
            val column = position % spanCount // item column
            if (includeEdge) {
                outRect.left =
                    spacing - column * spacing / spanCount // targetSpacing - column * ((1f / spanCount) * targetSpacing)
                outRect.right =
                    (column + 1) * spacing / spanCount // (column + 1) * ((1f / spanCount) * targetSpacing)

                if (position < spanCount) { // top edge
                    outRect.top = spacing
                }
                outRect.bottom = spacing // item bottom
            } else {
                outRect.left =
                    column * spacing / spanCount // column * ((1f / spanCount) * targetSpacing)
                outRect.right =
                    spacing - (column + 1) * spacing / spanCount // targetSpacing - (column + 1) * ((1f /    spanCount) * targetSpacing)
                if (position >= spanCount) {
                    outRect.top = spacing // item top
                }
            }
        }
    })
    return this
}