package com.liecoder.framework.manager

import android.app.Activity
import android.content.Context
import android.content.ContextWrapper
import android.os.Build
import android.util.Log
import com.liecoder.framework.ktx.e

/**
 * Activity管理类
 */
object ActivityManager {

    // 存储所有Activity实例的列表
    private val tasks = mutableListOf<Activity>()

    /**
     * 将一个新的Activity实例压入任务栈。
     * @param activity 要压入的Activity实例。
     */
    fun push(activity: Activity) {
        tasks.add(activity)
        e("push: ${activity::class.simpleName}")
    }

    /**
     * 从任务栈中弹出一个Activity实例。
     * @param activity 要弹出的Activity实例。
     */
    fun pop(activity: Activity) {
        tasks.remove(activity)
        e( "pop: ${activity::class.simpleName}")
    }

    /**
     * 获取任务栈顶的Activity实例。
     * @return 栈顶的Activity实例。
     */
    fun top(): Activity  = tasks.last()

    /**
     * 获取所有Activity实例的列表。
     * @return 包含所有Activity实例的列表。
     */
    fun getActivities(): MutableList<Activity>  = tasks

    /**
     * 结束所有Activity，并执行一个可选的回调。
     * @param block 结束所有Activity后执行的回调函数。
     */
    fun finishAllActivity(block: (() -> Unit)? = null) {
        val iterator = tasks.iterator()
        while (iterator.hasNext()) {
            val activity = iterator.next()
            iterator.remove()
            activity.finish()
        }
        block?.invoke()
    }

    /**
     * 结束除了指定类的所有Activity。
     * @param clazz 要保留的Activity的类。
     */
    fun finishOtherActivity(clazz: Class<out Activity>) {
        val iterator = tasks.iterator()
        while (iterator.hasNext()) {
            val activity = iterator.next()
            if (activity::class.java != clazz) {
                iterator.remove()
                activity.finish()
            }
        }
    }

    /**
     * 结束除了指定的Activity的所有其他Activity。
     * @param activities 要保留的Activity的类数组。
     */
    fun finishOtherActivity(vararg activities: Class<out Activity>) {
        val iterator = tasks.iterator()
        while (iterator.hasNext()) {
            val activity = iterator.next()
            if (!activities.contains(activity::class.java)) {
                iterator.remove()
                activity.finish()
            }
        }
    }

    /**
     * 结束指定的Activity。
     * @param clazz 要结束的Activity的类。
     */
    fun finishActivity(clazz: Class<out Activity>) {
        val iterator = tasks.iterator()
        while (iterator.hasNext()) {
            val activity = iterator.next()
            if (activity::class.java == clazz) {
                iterator.remove()
                activity.finish()
                break
            }
        }
    }

    /**
     * 结束指定的所有Activity。
     * @param activities 要结束的Activity的类数组。
     */
    fun finishActivity(vararg activities: Class<out Activity>) {
        val iterator = tasks.iterator()
        while (iterator.hasNext()) {
            val task = iterator.next()
            val activityIterator = activities.iterator()
            while (activityIterator.hasNext()) {
                val activity = activityIterator.next()
                if (task::class.java == activity) {
                    iterator.remove()
                    task.finish()
                    break
                }
            }
        }
    }

    /**
     * 检查指定的Activity是否存在于任务栈中。
     * @param clazz 要检查的Activity的类。
     * @return 如果存在返回true，否则返回false。
     */
    fun isActivityExistsInTask(clazz: Class<out Activity>): Boolean {
        if (clazz != null) {
            for (task in tasks) {
                if (task::class.java == clazz) {
                    return true
                }
            }
        }
        return false
    }

    /**
     * 从任务栈中获取指定的Activity。
     * @param clazz 要获取的Activity的类。
     * @return 如果找到则返回该Activity实例，否则返回null。
     */
    fun getActivity(clazz: Class<out Activity>): Activity? {
        if (clazz != null) {
            for (task in tasks) {
                if (task::class.java == clazz) {
                    return task
                }
            }
        }
        return null
    }

    /**
     * 检查Activity是否已经被销毁。
     * @param context 要检查的上下文。
     * @return 如果Activity已经被销毁或正在完成，则返回true，否则返回false。
     */
    fun isActivityDestroyed(context: Context): Boolean {
        val activity = findActivity(context)
        return if (activity != null) {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
                activity.isDestroyed || activity.isFinishing
            } else activity.isFinishing
        } else true
    }

    /**
     * 从给定的上下文中查找对应的Activity。
     * @param context 要查找的上下文。
     * @return 如果找到则返回对应的Activity实例，否则返回null。
     */
    private fun findActivity(context: Context): Activity? {
        if (context is Activity) {
            return context
        } else if (context is ContextWrapper) {
            return findActivity(context.baseContext)
        }
        return null
    }

}