package com.liecoder.framework.titlebar

import android.content.Context
import android.graphics.Color
import android.graphics.Typeface
import android.util.AttributeSet
import android.view.Gravity.BOTTOM
import android.view.Gravity.CENTER
import android.view.Gravity.CENTER_VERTICAL
import android.view.Gravity.END
import android.view.Gravity.LEFT
import android.view.Gravity.NO_GRAVITY
import android.view.Gravity.RIGHT
import android.view.Gravity.START
import android.view.View
import android.view.ViewGroup
import android.view.ViewGroup.LayoutParams.MATCH_PARENT
import android.view.ViewGroup.LayoutParams.WRAP_CONTENT
import android.widget.FrameLayout
import android.widget.TextView
import com.liecoder.framework.R
import com.liecoder.framework.ktx.ELLIPSIZE_NONE
import com.liecoder.framework.ktx.asTruncateAt
import com.liecoder.framework.ktx.asTypeface
import com.liecoder.framework.ktx.containsContent
import com.liecoder.framework.ktx.dpInt
import com.liecoder.framework.ktx.e
import com.liecoder.framework.ktx.getDrawableCompat
import kotlin.math.max


/**
 * Author: lieco(liecoder@163.com)
 * Date: 2024/8/18
 * Version: 1.0
 */
class TitleBar(context: Context, attrs: AttributeSet?, defStyleAttr: Int) :
    FrameLayout(context, attrs, defStyleAttr), View.OnLayoutChangeListener, View.OnClickListener {


    constructor(context: Context, attrs: AttributeSet?) : this(context, attrs, 0)

    constructor(context: Context) : this(context, null)

    /**
     *  初始化器
     */
    private val mDefaultStyle = DefaultStyle
    private var mCurrentStyle: ITitleBarStyleService

    /**
     *  点击事件监听器
     */
    private var mTitleBarClickListener: ITitleBarClickListener? = null

    /**
     *  标题栏子控件
     */
    internal var mLeftView: TextView
    internal var mTitleView: TextView
    internal var mRightView: TextView
    internal var mLineView: View

    /**
     *  控件水平间距
     */
    internal var mLeftHorizontalPadding: Int = 0
    internal var mTitleHorizontalPadding: Int = 0
    internal var mRightHorizontalPadding: Int = 0

    /**
     *  控件垂直间距
     */
    internal var mVerticalPadding: Int = 0

    /**
     *  图标大小尺寸
     */
    internal var mLeftIconWidth: Int = 24.dpInt
    internal var mLeftIconHeight: Int = 24.dpInt
    internal var mTitleIconWidth: Int = 24.dpInt
    internal var mTitleIconHeight: Int = 24.dpInt
    internal var mRightIconWidth: Int = 24.dpInt
    internal var mRightIconHeight: Int = 24.dpInt

    /**
     *  图标位置方向
     */
    internal var mLeftIconGravity: Int = LEFT
    internal var mTitleIconGravity: Int = RIGHT
    internal var mRightIconGravity: Int = RIGHT

    /**
     *  图标着色器
     */
    internal var mLeftIconTint: Int = Color.TRANSPARENT
    internal var mTitleIconTint: Int = Color.TRANSPARENT
    internal var mRightIconTint: Int = Color.TRANSPARENT


    init {
        val array = context.obtainStyledAttributes(
            attrs,
            R.styleable.TitleBar,
            0,
            R.style.TitleBarDefaultStyle
        )
        mCurrentStyle = when (R.styleable.TitleBar_barStyle) {
            0x10 -> StyleLight()
            0x20 -> StyleNight()
            0x30 -> StyleTransparent()
            0x40 -> StyleRipple()
            else -> mDefaultStyle
        }
        mTitleView = mCurrentStyle.createTitleView(context).apply {
            layoutParams = LayoutParams(WRAP_CONTENT, MATCH_PARENT, CENTER)
        }
        mLeftView = mCurrentStyle.createLeftView(context).apply {
            layoutParams = LayoutParams(WRAP_CONTENT, MATCH_PARENT, START or CENTER_VERTICAL)
        }
        mRightView = mCurrentStyle.createRightView(context).apply {
            layoutParams = LayoutParams(WRAP_CONTENT, MATCH_PARENT, END or CENTER_VERTICAL)
        }
        mLineView = mCurrentStyle.createLineView(context).apply {
            layoutParams = LayoutParams(MATCH_PARENT, mCurrentStyle.getLineSize(context), BOTTOM)
        }

        val titleIconGravity = mCurrentStyle.getTitleIconGravity(context)
        setTitleIconGravity(array.getInt(R.styleable.TitleBar_titleIconGravity, titleIconGravity))
        val leftIconGravity = mCurrentStyle.getLeftIconGravity(context)
        setLeftIconGravity(array.getInt(R.styleable.TitleBar_leftIconGravity, leftIconGravity))
        val rightIconGravity = mCurrentStyle.getRightIconGravity(context)
        setRightIconGravity(array.getInt(R.styleable.TitleBar_rightIconGravity, rightIconGravity))

        val titleIconWidth = mCurrentStyle.getTitleIconWidth(context)
        val titleIconWidthPixel =
            array.getDimensionPixelSize(R.styleable.TitleBar_titleIconWidth, titleIconWidth)
        val titleIconHeight = mCurrentStyle.getTitleIconHeight(context)
        val titleIconHeightPixel =
            array.getDimensionPixelSize(R.styleable.TitleBar_titleIconHeight, titleIconHeight)
        setTitleIconSize(titleIconWidthPixel, titleIconHeightPixel)
        val leftIconWidth = mCurrentStyle.getLeftIconWidth(context)
        val leftIconWidthPixel =
            array.getDimensionPixelSize(R.styleable.TitleBar_leftIconWidth, leftIconWidth)
        val leftIconHeight = mCurrentStyle.getLeftIconHeight(context)
        val leftIconHeightPixel =
            array.getDimensionPixelSize(R.styleable.TitleBar_leftIconHeight, leftIconHeight)
        setLeftIconSize(leftIconWidthPixel, leftIconHeightPixel)
        val rightIconWidth = mCurrentStyle.getRightIconWidth(context)
        val rightIconWidthPixel =
            array.getDimensionPixelSize(R.styleable.TitleBar_rightIconWidth, rightIconWidth)
        val rightIconHeight = mCurrentStyle.getRightIconHeight(context)
        val rightIconHeightPixel =
            array.getDimensionPixelSize(R.styleable.TitleBar_rightIconHeight, rightIconHeight)
        setRightIconSize(rightIconWidthPixel, rightIconHeightPixel)

        val titleIconPadding = mCurrentStyle.getTitleIconPadding(context)
        val titleIconPaddingPixel =
            array.getDimensionPixelSize(R.styleable.TitleBar_titleIconPadding, titleIconPadding)
        setTitleIconPadding(titleIconPaddingPixel)
        val leftIconPadding = mCurrentStyle.getLeftIconPadding(context)
        val leftIconPaddingPixel =
            array.getDimensionPixelSize(R.styleable.TitleBar_leftIconPadding, leftIconPadding)
        setLeftIconPadding(leftIconPaddingPixel)
        val rightIconPadding = mCurrentStyle.getRightIconPadding(context)
        val rightIconPaddingPixel =
            array.getDimensionPixelSize(R.styleable.TitleBar_rightIconPadding, rightIconPadding)
        setRightIconPadding(rightIconPaddingPixel)

        val strPlaceholder = R.string.bar_string_placeholder
        if (array.hasValue(R.styleable.TitleBar_title)) {
            if (array.getResourceId(R.styleable.TitleBar_title, 0) == strPlaceholder) {
                mTitleView.text = mCurrentStyle.getTitle(context)
            } else {
                mTitleView.text = array.getString(R.styleable.TitleBar_title)
            }
        }
        if (array.hasValue(R.styleable.TitleBar_leftTitle)) {
            if (array.getResourceId(R.styleable.TitleBar_leftTitle, 0) == strPlaceholder) {
                mLeftView.text = mCurrentStyle.getLeftTitle(context)
            } else {
                mLeftView.text = array.getString(R.styleable.TitleBar_leftTitle)
            }
        }
        if (array.hasValue(R.styleable.TitleBar_rightTitle)) {
            if (array.getResourceId(R.styleable.TitleBar_rightTitle, 0) == strPlaceholder) {
                mRightView.text = mCurrentStyle.getRightTitle(context)
            } else {
                mRightView.text = array.getString(R.styleable.TitleBar_rightTitle)
            }
        }

        if (array.hasValue(R.styleable.TitleBar_titleIconTint))
            setTitleIconTint(array.getColor(R.styleable.TitleBar_titleIconTint, 0))
        if (array.hasValue(R.styleable.TitleBar_leftIconTint))
            setLeftIconTint(array.getColor(R.styleable.TitleBar_leftIconTint, 0))
        if (array.hasValue(R.styleable.TitleBar_rightIconTint))
            setRightIconTint(array.getColor(R.styleable.TitleBar_rightIconTint, 0))

        val bgPlaceholderId = R.drawable.bar_drawable_placeholder
        if (array.hasValue(R.styleable.TitleBar_titleIcon)) {
            val titleIconId = array.getResourceId(R.styleable.TitleBar_titleIcon, 0)
            setTitleIcon(context.getDrawableCompat(titleIconId))
        }
        if (array.hasValue(R.styleable.TitleBar_leftIcon)) {
            val leftIconId = array.getResourceId(R.styleable.TitleBar_leftIcon, 0)
            if (leftIconId == bgPlaceholderId) setLeftIcon(mCurrentStyle.getBackButtonDrawable(context))
            else
                setLeftIcon(context.getDrawableCompat(leftIconId))
        }
        if (array.hasValue(R.styleable.TitleBar_rightIcon)) {
            val rightIconId = array.getResourceId(R.styleable.TitleBar_rightIcon, 0)
            setRightIcon(context.getDrawableCompat(rightIconId))
        }

        setTitleColor(
            if (array.hasValue(R.styleable.TitleBar_titleColor))
                array.getColorStateList(R.styleable.TitleBar_titleColor)
            else
                mCurrentStyle.getTitleColor(context)
        )
        setLeftTitleColor(
            if (array.hasValue(R.styleable.TitleBar_leftTitleColor))
                array.getColorStateList(R.styleable.TitleBar_leftTitleColor)
            else
                mCurrentStyle.getLeftTitleColor(context)
        )
        setRightTitleColor(
            if (array.hasValue(R.styleable.TitleBar_rightTitleColor))
                array.getColorStateList(R.styleable.TitleBar_rightTitleColor)
            else
                mCurrentStyle.getRightTitleColor(context)
        )

        setTitleSize(
            size = if (array.hasValue(R.styleable.TitleBar_titleSize))
                array.getDimensionPixelSize(R.styleable.TitleBar_titleSize, 0).toFloat()
            else
                mCurrentStyle.getTitleSize(context)
        )
        setLeftTitleSize(
            size = if (array.hasValue(R.styleable.TitleBar_leftTitleSize))
                array.getDimensionPixelSize(R.styleable.TitleBar_leftTitleSize, 0).toFloat()
            else
                mCurrentStyle.getLeftTitleSize(context)
        )
        setRightTitleSize(
            size = if (array.hasValue(R.styleable.TitleBar_rightTitleSize))
                array.getDimensionPixelSize(R.styleable.TitleBar_rightTitleSize, 0).toFloat()
            else
                mCurrentStyle.getRightTitleSize(context)
        )
        val titleStyle = if (array.hasValue(R.styleable.TitleBar_titleStyle))
            array.getInt(R.styleable.TitleBar_titleStyle, Typeface.NORMAL)
        else
            mCurrentStyle.getTitleStyle(context)
        setTitleStyle(titleStyle.asTypeface, titleStyle)
        val leftTitleStyle = if (array.hasValue(R.styleable.TitleBar_leftTitleStyle))
            array.getInt(R.styleable.TitleBar_leftTitleStyle, Typeface.NORMAL)
        else
            mCurrentStyle.getLeftTitleStyle(context)
        setLeftTitleStyle(leftTitleStyle.asTypeface, leftTitleStyle)
        val rightTitleStyle = if (array.hasValue(R.styleable.TitleBar_rightTitleStyle))
            array.getInt(R.styleable.TitleBar_rightTitleStyle, Typeface.NORMAL)
        else
            mCurrentStyle.getRightTitleStyle(context)
        setRightTitleStyle(rightTitleStyle.asTypeface, rightTitleStyle)

        val titleOverflowMode = if (array.hasValue(R.styleable.TitleBar_titleOverflowMode))
            array.getInt(R.styleable.TitleBar_titleOverflowMode, ELLIPSIZE_NONE).asTruncateAt
        else
            mCurrentStyle.getTitleOverflowMode(context)
        setTitleOverflowMode(titleOverflowMode)
        val leftTitleOverflowMode = if (array.hasValue(R.styleable.TitleBar_leftTitleOverflowMode))
            array.getInt(R.styleable.TitleBar_leftTitleOverflowMode, ELLIPSIZE_NONE).asTruncateAt
        else
            mCurrentStyle.getLeftTitleOverflowMode(context)
        setLeftTitleOverflowMode(leftTitleOverflowMode)
        val rightTitleOverflowMode =
            if (array.hasValue(R.styleable.TitleBar_rightTitleOverflowMode))
                array.getInt(R.styleable.TitleBar_rightTitleOverflowMode, 0).asTruncateAt
            else
                mCurrentStyle.getRightTitleOverflowMode(context)
        setRightTitleOverflowMode(rightTitleOverflowMode)

        if (array.hasValue(R.styleable.TitleBar_titleGravity))
            setTitleGravity(array.getInt(R.styleable.TitleBar_titleGravity, NO_GRAVITY))

        if (array.hasValue(R.styleable.TitleBar_android_background)) {
            if (array.getResourceId(R.styleable.TitleBar_android_background, 0) == bgPlaceholderId)
                background = mCurrentStyle.getTitleBarBackground(context)
        }
        if (array.hasValue(R.styleable.TitleBar_leftBackground)) {
            if (array.getResourceId(R.styleable.TitleBar_leftBackground, 0) == bgPlaceholderId) {
                setLeftBackground(mCurrentStyle.getLeftTitleBackground(context))
            } else {
                setLeftBackground(array.getDrawable(R.styleable.TitleBar_leftBackground))
            }
        }
        if (array.hasValue(R.styleable.TitleBar_rightBackground)) {
            if (array.getResourceId(R.styleable.TitleBar_rightBackground, 0) == bgPlaceholderId) {
                setRightBackground(mCurrentStyle.getRightTitleBackground(context))
            } else {
                setRightBackground(array.getDrawable(R.styleable.TitleBar_rightBackground))
            }
        }

        if (array.hasValue(R.styleable.TitleBar_leftForeground)) {
            if (array.getResourceId(R.styleable.TitleBar_leftForeground, 0) == bgPlaceholderId) {
                setLeftForeground(mCurrentStyle.getLeftTitleForeground(context))
            } else {
                setLeftForeground(array.getDrawable(R.styleable.TitleBar_leftForeground))
            }
        }
        if (array.hasValue(R.styleable.TitleBar_rightForeground)) {
            if (array.getResourceId(R.styleable.TitleBar_rightForeground, 0) == bgPlaceholderId) {
                setRightForeground(mCurrentStyle.getRightTitleForeground(context))
            } else {
                setRightForeground(array.getDrawable(R.styleable.TitleBar_rightForeground))
            }
        }

        setLineVisible(
            array.getBoolean(
                R.styleable.TitleBar_lineDrawable,
                mCurrentStyle.isLineVisible(context)
            )
        )

        if (array.hasValue(R.styleable.TitleBar_lineDrawable)) {
            if (array.getResourceId(R.styleable.TitleBar_lineDrawable, 0) == bgPlaceholderId) {
                setLineDrawable(mCurrentStyle.getLineDrawable(context))
            } else {
                setLineDrawable(array.getDrawable(R.styleable.TitleBar_lineDrawable))
            }
        }

        if (array.hasValue(R.styleable.TitleBar_lineSize))
            setLineSize(array.getDimensionPixelSize(R.styleable.TitleBar_lineSize, 0))

        setChildHorizontalPadding(
            leftHorizontalPadding = array.getDimensionPixelSize(
                R.styleable.TitleBar_leftHorizontalPadding,
                mCurrentStyle.getLeftHorizontalPadding(context)
            ),
            titleHorizontalPadding = array.getDimensionPixelSize(
                R.styleable.TitleBar_titleHorizontalPadding,
                mCurrentStyle.getTitleHorizontalPadding(context)
            ),
            rightHorizontalPadding = array.getDimensionPixelSize(
                R.styleable.TitleBar_rightHorizontalPadding,
                mCurrentStyle.getRightHorizontalPadding(context)
            )
        )
        setChildVerticalPadding(
            array.getDimensionPixelSize(
                R.styleable.TitleBar_childVerticalPadding,
                mCurrentStyle.getChildVerticalPadding(context)
            )
        )

        array.recycle()

        addView(mTitleView, 0)
        addView(mLeftView, 1)
        addView(mRightView, 2)
        addView(mLineView, 3)

        addOnLayoutChangeListener(this)
    }

    override fun onMeasure(widthMeasureSpec: Int, heightMeasureSpec: Int) {
        super.onMeasure(widthMeasureSpec, heightMeasureSpec)

        val titleBarWidth = this@TitleBar.measuredWidth
        val titleBarHeight = this@TitleBar.measuredHeight

        val leftViewMeasuredWidth = mLeftView.measuredWidth
        val leftViewMeasuredHeight = mLeftView.measuredHeight

        val titleViewMeasuredWidth = mTitleView.measuredWidth
        val titleViewMeasuredHeight = mTitleView.measuredHeight

        val rightViewMeasuredWidth = mRightView.measuredWidth
        val rightViewMeasuredHeight = mRightView.measuredHeight

        val maxEdgeWidth = max(leftViewMeasuredWidth, rightViewMeasuredWidth)
        val calculateTotalWidth = maxEdgeWidth * 2 + titleViewMeasuredWidth

        if (calculateTotalWidth <= titleBarWidth) {
            return measureTitleBar(
                leftViewWidth = (if (mLeftView.containsContent) leftViewMeasuredWidth else 0),
                titleViewWidth = titleViewMeasuredWidth,
                rightViewWidth = (if (mRightView.containsContent) rightViewMeasuredWidth else 0),
                titleBarHeightMeasureSpec = heightMeasureSpec
            )
        }
        val leftViewWidth: Int
        val titleViewWidth: Int
        val rightViewWidth: Int
        if (maxEdgeWidth > titleBarWidth / 3) {
            leftViewWidth = titleBarWidth / 4
            titleViewWidth = titleBarWidth / 2
            rightViewWidth = titleBarWidth / 4
        } else {
            leftViewWidth = maxEdgeWidth
            titleViewWidth = titleBarWidth - maxEdgeWidth * 2
            rightViewWidth = maxEdgeWidth
        }
        measureTitleBar(
            leftViewWidth = (if (mLeftView.containsContent) leftViewWidth else 0),
            titleViewWidth = titleViewWidth,
            rightViewWidth = (if (mRightView.containsContent) rightViewWidth else 0),
            titleBarHeightMeasureSpec = heightMeasureSpec
        )
    }

    private fun measureTitleBar(
        leftViewWidth: Int,
        titleViewWidth: Int,
        rightViewWidth: Int,
        titleBarHeightMeasureSpec: Int,
    ) {
        val leftWidthMeasureSpec = MeasureSpec.makeMeasureSpec(leftViewWidth, MeasureSpec.EXACTLY)
        val titleWidthMeasureSpec = MeasureSpec.makeMeasureSpec(titleViewWidth, MeasureSpec.EXACTLY)
        val rightWidthMeasureSpec = MeasureSpec.makeMeasureSpec(rightViewWidth, MeasureSpec.EXACTLY)
        measureChildWithMargins(mLeftView, leftWidthMeasureSpec, 0, titleBarHeightMeasureSpec, 0)
        measureChildWithMargins(mTitleView, titleWidthMeasureSpec, 0, titleBarHeightMeasureSpec, 0)
        measureChildWithMargins(mRightView, rightWidthMeasureSpec, 0, titleBarHeightMeasureSpec, 0)
        val titleBarMeasuredHeight = this.measuredHeight
        if (titleBarMeasuredHeight != mLeftView.measuredHeight)
            mLeftView.measure(
                leftWidthMeasureSpec,
                MeasureSpec.makeMeasureSpec(titleBarMeasuredHeight, MeasureSpec.EXACTLY)
            )

        if (titleBarMeasuredHeight != mTitleView.measuredHeight)
            mTitleView.measure(
                titleWidthMeasureSpec,
                MeasureSpec.makeMeasureSpec(titleBarMeasuredHeight, MeasureSpec.EXACTLY)
            )

        if (titleBarMeasuredHeight != mRightView.measuredHeight)
            mRightView.measure(
                rightWidthMeasureSpec,
                MeasureSpec.makeMeasureSpec(titleBarMeasuredHeight, MeasureSpec.EXACTLY)
            )
    }

    override fun onLayoutChange(
        p0: View?,
        p1: Int,
        p2: Int,
        p3: Int,
        p4: Int,
        p5: Int,
        p6: Int,
        p7: Int,
        p8: Int
    ) {
        if (!mLeftView.isClickable) mLeftView.isClickable = true
        if (!mTitleView.isClickable) mTitleView.isClickable = true
        if (!mRightView.isClickable) mRightView.isClickable = true
        mLeftView.isEnabled = mLeftView.containsContent
        mTitleView.isEnabled = mTitleView.containsContent
        mRightView.isEnabled = mRightView.containsContent
    }

    override fun setLayoutParams(params: ViewGroup.LayoutParams?) {
        if (params?.width == WRAP_CONTENT) params.width = MATCH_PARENT
        setChildVerticalPadding(if (params?.height == WRAP_CONTENT) mVerticalPadding else 0)
        super.setLayoutParams(params)
    }

    override fun generateDefaultLayoutParams(): LayoutParams =
        LayoutParams(MATCH_PARENT, WRAP_CONTENT)

    fun setOnTitleBarClickListener(listener: ITitleBarClickListener): TitleBar {
        mTitleBarClickListener = listener
        mLeftView.setOnClickListener(this)
        mTitleView.setOnClickListener(this)
        mRightView.setOnClickListener(this)
        return this
    }

    override fun onClick(view: View?) {
        when (view) {
            mLeftView -> mTitleBarClickListener?.onLeftClick(this)
            mTitleView -> mTitleBarClickListener?.onTitleClick(this)
            mRightView -> mTitleBarClickListener?.onRightClick(this)
        }
    }

}