package com.liecoder.framework.base

import android.annotation.SuppressLint
import android.content.pm.ActivityInfo
import android.graphics.Color
import android.graphics.drawable.Drawable
import android.os.Bundle
import android.view.Window
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleEventObserver
import androidx.lifecycle.LifecycleOwner
import com.liecoder.framework.ktx.i
import com.liecoder.framework.dialog.LoadingView
import com.liecoder.framework.titlebar.ITitleBarClickListener
import com.liecoder.framework.titlebar.TitleBar
import com.liecoder.framework.toast.TipsToast
import com.liecoder.framework.toast.TipsToast.Errors
import com.liecoder.framework.toast.TipsToast.Infos
import com.liecoder.framework.toast.TipsToast.Success
import com.liecoder.framework.toast.TipsToast.Warnings
import com.liecoder.framework.toast.ToastStyle
import com.liecoder.framework.utils.StatusBarUtils
import com.yinglan.keyboard.HideUtil
import net.yslibrary.android.keyboardvisibilityevent.KeyboardVisibilityEvent
import net.yslibrary.android.keyboardvisibilityevent.KeyboardVisibilityEventListener
import net.yslibrary.android.keyboardvisibilityevent.Unregistrar

/**
 * 基础活动类，提供通用的活动设置和功能。
 *
 * 此类继承自 BasePermissionActivity，并实现了多个接口，以提供权限处理、生命周期观察、标题栏点击事件、键盘可见性事件等功能。
 *
 * @author liecoder
 * @date 2024/6/20 周四
 * @version 1.0
 */
abstract class BaseActivity : BasePermissionActivity(), IBaseService, LifecycleEventObserver,
    ITitleBarClickListener, KeyboardVisibilityEventListener {

    // 用于取消键盘可见性事件监听的 Unregistrar 对象。
    private var unregistrar: Unregistrar? = null


    // 标记键盘是否可见。
    var keyBoardVisible: Boolean = false


    // 延迟初始化Loading框。
    private val mLoadingView: LoadingView by lazy { LoadingView(mContext = this) }


    /**
     * 活动创建时的回调方法。
     *
     * 在这个方法中，执行一系列的初始化操作，包括设置窗口特性、请求屏幕方向、设置内容视图、初始化、设置视图、加载数据、设置状态栏、注册生命周期观察者、初始化隐藏工具和注册键盘可见性事件监听。
     *
     * @param savedInstanceState 保存的实例状态。
     */
    @SuppressLint("SourceLockedOrientationActivity")
    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        requestWindowFeature(Window.FEATURE_NO_TITLE)
        requestedOrientation = ActivityInfo.SCREEN_ORIENTATION_PORTRAIT
        setContentView()
        init()
        setupViews()
        loadData()
        setupStatusBar()
        lifecycle.addObserver(this)
        HideUtil.init(this)
        unregistrar = KeyboardVisibilityEvent.registerEventListener(
            this, this
        )
    }

    override fun init() {
    }

    override fun setupViews() {
    }

    override fun loadData() {
    }

    override fun setupStatusBar() = StatusBarUtils.setStatusBar(
        activity = this,
        darkContent = statusBarDarkContent(),
        statusBarColor = statusBarColor(),
        translucent = statusBarTranslucent()
    )

    override fun statusBarDarkContent(): Boolean = true

    override fun statusBarColor(): Int = Color.WHITE

    override fun statusBarTranslucent(): Boolean = false

    override fun onStateChanged(
        source: LifecycleOwner, event: Lifecycle.Event
    ) {
        i("ActivityStateChanged: $event")
        if (event == Lifecycle.Event.ON_DESTROY) unregistrar?.unregister()
    }

    override fun onVisibilityChanged(isOpen: Boolean) {
        i("KeyboardStateChanged: isVisible = $isOpen")
        keyBoardVisible = isOpen
    }

    override fun onLeftClick(titleBar: TitleBar) {
        super.onLeftClick(titleBar)
        finish()
    }

    override fun onTitleClick(titleBar: TitleBar) {
        super.onTitleClick(titleBar)
    }

    override fun onRightClick(titleBar: TitleBar) {
        super.onRightClick(titleBar)
    }

    override fun showLoading(
        show: Boolean, message: String?
    ) {
        if (!show) return hideLoading()
        mLoadingView.show(message = message)
    }

    override fun hideLoading() = mLoadingView.dismiss()

    override fun success(message: CharSequence?) = Success.show(message)

    override fun errors(message: CharSequence?) = Errors.show(message)

    override fun warnings(message: CharSequence?) = Warnings.show(message)

    override fun info(message: CharSequence?) = Infos.show(message)

    override fun custom(
        message: CharSequence?,
        successIcon: Drawable?,
        errorIcon: Drawable?,
        warningIcon: Drawable?,
        gravity: Int,
        xOffset: Int,
        yOffset: Int,
        duration: Int,
        style: ToastStyle?
    ) = TipsToast.custom(
        message = message,
        successIcon = successIcon,
        errorIcon = errorIcon,
        warningIcon = warningIcon,
        gravity = gravity,
        xOffset = xOffset,
        yOffset = yOffset,
        duration = duration,
        style = style
    )

}