package com.liecoder.framework.ktx

import android.content.Intent
import android.graphics.Bitmap
import android.net.Uri
import android.os.Environment
import com.liecoder.framework.global.AppHelper.application
import com.liecoder.framework.model.DateFormatPattern
import java.io.File
import java.io.FileOutputStream
import java.util.Date


/**
 * 将 Bitmap 对象保存到系统相册。
 * 请确保在调用此函数之前 [application] 已经被初始化，因为它依赖于应用程序的上下文来发送广播通知系统图片已更新。
 * @param imageType 压缩格式，如 PNG 或 JPEG。
 * @param compressQuality 压缩质量，从0（最差）到100（无损）。
 * @param album 相册名称，用于在公共图片目录下创建子目录。
 * @param imageFileName 要保存的图片文件名，如果为 null，则使用当前日期和时间。
 * @return 保存的图片文件的绝对路径，如果保存失败返回 null。
 */
fun Bitmap.saveAs(
    imageType: Bitmap.CompressFormat = Bitmap.CompressFormat.PNG,
    compressQuality: Int = 100,
    album: String = "Default",
    imageFileName: String? = null
): String? {
    // 检查外部存储是否可用
    if (Environment.getExternalStorageState() != Environment.MEDIA_MOUNTED) return null
    // 获取或创建相册目录
    val albumFile = File(
        Environment.getExternalStoragePublicDirectory(Environment.DIRECTORY_PICTURES),
        album
    )
    // 如果无法创建目录，返回 null
    if (!albumFile.exists() && !albumFile.mkdirs()) return null
    // 生成文件名，如果未提供自定义文件名，则使用当前日期和时间
    val fileName =
        imageFileName ?: Date().format(DateFormatPattern.FULL_DATE_TIME_DASH) ?: return null
    val imageFile = File(albumFile, fileName)
    try {
        // 使用 FileOutputStream 将图片写入文件
        val outputStream = FileOutputStream(imageFile)
        compress(imageType, compressQuality, outputStream)
        outputStream.flush() // 确保所有数据都写入到文件
        outputStream.close() // 关闭流
        // 媒体扫描器，通知系统图片已更新
        val mediaScannerIntent = Intent(Intent.ACTION_MEDIA_SCANNER_SCAN_FILE)
        val contentUri = Uri.fromFile(imageFile)
        mediaScannerIntent.data = contentUri
        application.sendBroadcast(mediaScannerIntent)
        return imageFile.absolutePath // 返回图片的完整路径
    } catch (e: Exception) {
        e.printStackTrace()
        return null
    }

}
