package com.liecoder.framework.ktx

import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleEventObserver
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.ViewModel
import androidx.lifecycle.ViewModelLazy
import androidx.lifecycle.ViewModelProvider
import androidx.lifecycle.ViewModelStore
import androidx.lifecycle.ViewModelStoreOwner

/**
 * 用于 Android 应用程序中的自定义 ViewModel 共享机制。该机制允许开发者在不同的生命周期宿主（LifecycleOwner）之
 * 间共享 ViewModel 实例。
 * 要使用自定义的 ViewModel 共享机制，请按照以下步骤操作：
 * 通过 shareViewModels 函数获取延迟初始化的 ViewModel 实例。
 * ```kotlin
 * val myViewModel: MyViewModel by activity.shareViewModels(<ScopeName>)
 * ```
 */


/**
 *  存储作用域名称与对应的 VMStore 实例的映射。
 */
val vMStores = HashMap<String, VMStore>()


/**
 *  提供一个延迟初始化的 ViewModel 实例，该实例可以在指定的作用域内被共享。
 *  @param scopeName 作用域名称，用于区分不同的 ViewModel 存储。
 *  @param factory 可选的 ViewModelProvider 工厂，用于创建 ViewModel 实例。
 *  @return  Lazy<VM>: 延迟初始化的 ViewModel 实例。
 */
inline fun <reified VM : ViewModel> LifecycleOwner.shareViewModels(
    scopeName: String,
    factory: ViewModelProvider.Factory? = null
): Lazy<VM> {
    val store: VMStore
    if (vMStores.keys.contains(scopeName)) {
        store = vMStores[scopeName]!!
    } else {
        store = VMStore()
        vMStores[scopeName] = store
    }
    store.register(this)
    return ViewModelLazy(VM::class, { store.viewModelStore },
        { factory ?: ViewModelProvider.NewInstanceFactory() })
}


class VMStore : ViewModelStoreOwner {

    // 存储所有注册的生命周期宿主。
    private val bindTargets = ArrayList<LifecycleOwner>()

    // 存储 ViewModel 实例的 ViewModelStore，延迟初始化。
    private var vmStore: ViewModelStore? = null

    /**
     *  注册一个生命周期宿主到当前 VMStore。如果宿主尚未注册，则添加到 bindTargets 并监听其生命周期事件。当宿主销
     *  毁时，从 bindTargets 中移除，并清理相关资源。
     *  @param host  要注册的生命周期宿主。
     */
    fun register(host: LifecycleOwner) {
        if (!bindTargets.contains(host)) {
            bindTargets.add(host)
            host.lifecycle.addObserver(object : LifecycleEventObserver {
                override fun onStateChanged(
                    source: LifecycleOwner,
                    event: Lifecycle.Event
                ) {
                    if (event == Lifecycle.Event.ON_DESTROY) {
                        host.lifecycle.removeObserver(this)
                        bindTargets.remove(host)
                        if (bindTargets.isEmpty()) {
                            vMStores.entries.find { it.value == this@VMStore }?.also {
                                vMStores.clear()
                                vMStores.remove(it.key)
                            }
                        }
                    }
                }
            })
        }
    }

    /**
     *  获取与当前 VMStore 关联的 ViewModelStore 实例。
     *  如果 vmStore 尚未初始化，则创建一个新的 ViewModelStore 实例。
     *  @return ViewModelStore: 与当前 VMStore 关联的 ViewModelStore 实例。
     */
    override fun getViewModelStore(): ViewModelStore {
        if (vmStore == null) vmStore = ViewModelStore()
        return vmStore!!
    }

}