package com.liecoder.framework.titlebar

import android.content.Context
import android.graphics.drawable.Drawable
import android.util.TypedValue
import androidx.core.content.res.ResourcesCompat


/**
 * Author: lieco(liecoder@163.com)
 * Date: 2024/8/18
 * Version: 1.0
 */
class StyleRipple : StyleTransparent() {

    /**
     * 获取标题栏左侧背景的 Drawable 对象。
     *
     * @param context 上下文对象。
     * @return 标题栏左侧背景的 Drawable 对象，可能为 null。
     */
    override fun getLeftTitleBackground(context: Context): Drawable {
        return createRippleDrawable(context) ?: super.getLeftTitleBackground(context)
    }

    /**
     * 获取标题栏右侧背景的 Drawable 对象。
     *
     * @param context 上下文对象。
     * @return 标题栏右侧背景的 Drawable 对象，可能为 null。
     */
    override fun getRightTitleBackground(context: Context): Drawable {
        return createRippleDrawable(context) ?: super.getRightTitleBackground(context)
    }

    /**
     * 创建一个 Ripple 效果的 Drawable。
     *
     * 此函数尝试从给定的上下文 [context] 的主题中解析一个无边框的选择响应 Drawable 属性（[android.R.attr.selectableItemBackgroundBorderless]）。
     * 如果找到该属性，将使用 [ContextCompat.getDrawableCompat] 方法获取对应的 Drawable 对象。
     *
     * @param context 上下文对象，用于访问主题和资源。
     * @return 返回解析得到的 Drawable 对象，如果没有找到对应的属性则返回 null。
     */
    private fun createRippleDrawable(context: Context): Drawable? {
        // 创建 TypedValue 对象以接收主题属性值
        val typedValue = TypedValue()
        // 使用 resolveAttribute 解析主题属性，第三个参数 true 表示允许返回资源引用
        val resolveAttribute = context.theme.resolveAttribute(
            android.R.attr.selectableItemBackgroundBorderless,
            typedValue, true
        )
        // 如果成功解析，使用 ContextCompat 的 getDrawableCompat 方法获取 Drawable
        // 如果没有找到对应的主题属性，返回 null
        return if (resolveAttribute) ResourcesCompat.getDrawable(context.resources, typedValue.resourceId, context.theme) else null

    }

}