package com.liecoder.framework.base

import androidx.activity.result.ActivityResultLauncher
import com.liecoder.framework.model.Permission

/**
 * 权限服务接口，用于统一管理和处理应用程序的权限请求。
 *
 * 此接口定义了一组方法，用于注册权限请求的回调、处理权限请求逻辑、显示权限说明对话框、引导用户前往权限设置页面，以及获取和解释权限请求的相关信息。
 * 通过实现此接口，应用程序可以在不同组件中保持一致的权限处理策略。
 *
 * @author liecoder
 * @date 2024/6/21 周五
 * @version 1.0
 */
interface IPermissionService {

    /**
     * 注册一个用于请求运行时权限的 [ActivityResultLauncher]。
     *
     * 此方法应返回一个配置好的 [ActivityResultLauncher] 实例，用于在需要时触发权限请求。
     *
     * @return 配置好的 [ActivityResultLauncher] 实例，其结果用于处理权限请求的结果。
     */
    fun registerPermissionLauncher(): ActivityResultLauncher<String>

    /**
     * 处理权限请求逻辑。
     *
     * 根据用户对权限请求的响应，执行不同的逻辑：
     *
     *     如果权限已授予，则调用 [onPermissionGranted] 回调。
     *     如果权限被拒绝且不再提示，则调用 [onPermissionDenied] 回调（如果提供）。
     *     如果需要显示权限说明对话框，则应由实现类负责显示。
     *
     *
     * @param permission 请求的权限。
     * @param onPermissionGranted 权限授予时调用的回调。
     * @param onPermissionDenied 权限被拒绝时调用的回调，可为空。
     */
    fun withPermission(
        permission: Permission,
        onPermissionGranted: () -> Unit,
        onPermissionDenied: (() -> Unit)? = null
    )


    /**
     * 显示权限请求的说明对话框。
     *
     * 当用户已拒绝过该权限请求，且再次请求时需要额外解释的情况下，应显示此对话框。
     *
     * @param permission 需要解释的权限。
     */
    fun showRequestPermissionRationaleDialog(permission: Permission)

    /**
     * 引导用户前往应用的权限设置页面。
     *
     * 此方法应触发导航至应用的权限设置页面，以便用户可以手动授予权限。
     */
    fun grantingPermissions()

    /**
     * 获取为什么需要特定权限的解释说明。
     *
     * 此方法返回一个字符串，解释为什么应用需要请求的特定权限。
     *
     * @param permission 请求解释的权限。
     * @return 返回权限请求的解释说明。
     */
    fun getExplanationForSensitiveInformation(permission: Permission): String?

    /**
     * 解释对敏感信息的访问权限。
     *
     * 此方法提供额外的解释或步骤，以便用户理解为什么应用需要访问敏感信息。
     *
     * @param permission 需要访问的敏感权限。
     */
    fun explainAccessToMoreSensitiveInformation(permission: Permission)

}