package com.liecoder.framework.dispatcher

import android.os.Handler
import android.os.Looper
import androidx.activity.ComponentActivity
import androidx.activity.OnBackPressedCallback
import com.liecoder.framework.toast.TipsToast.Infos
import kotlin.system.exitProcess


/**
 * 应用程序退出分发器，用于管理基于返回按钮的应用程序退出逻辑。
 *
 * 此对象提供了一种标准化的方式来处理应用程序的退出，通过注册到 Activity 的返回按钮分发器。
 * 它使用一个延迟操作来确定用户是否真的想要退出应用程序，避免因误操作而退出。
 */
object AppExitDispatcher {

    /**
     * 标记是否已经向用户显示过退出警告。
     *
     * 这个变量用来跟踪是否已经向用户显示了退出警告的提示。如果用户在提示后的指定时间内再次按下返回按钮，则应用程序将退出。
     */
    private var hasWarnedAboutExit = false


    /**
     * 在主线程上运行的 Handler 对象，用于处理退出延迟逻辑。
     *
     * 这个 Handler 用于在用户按下返回按钮后创建一个延迟，如果在延迟期间没有再次按下返回按钮，
     * 则重置退出警告状态并取消退出操作。
     */
    private val handler = Handler(Looper.getMainLooper()) {
        hasWarnedAboutExit = false
        return@Handler true
    }


    /**
     * 注册退出逻辑到 Activity 的返回按钮分发器。
     *
     * 此方法使用 Activity 的 onBackPressedDispatcher 来添加一个回调，用于处理返回按钮的按下事件。
     * 如果用户在短时间内连续按下返回按钮，则会执行退出逻辑。
     *
     * @param activity 当前的 Activity 实例。
     * @param exitBlock 用户自定义的退出前执行的代码块，可以为 null。
     */
    fun register(activity: ComponentActivity, exitBlock: (() -> Unit)? = null) =
        activity.onBackPressedDispatcher.addCallback(
            activity,object : OnBackPressedCallback(true) {

                override fun handleOnBackPressed() {
                    if (!hasWarnedAboutExit) {
                        hasWarnedAboutExit = true
                        Infos.show("再按一次返回键退出APP")
                        handler.sendEmptyMessageDelayed(0, 2_000)
                    } else {
                        exitBlock?.invoke()
                        activity.finish()
                        exitProcess(0)
                    }
                }

            }
        )

}
