package com.liecoder.framework.ktx

import android.app.Activity
import android.content.Context
import android.content.Intent
import android.os.Build
import android.os.Bundle
import android.view.View
import androidx.core.app.ActivityCompat
import com.liecoder.framework.global.AppHelper.application


/**
 * 判断当前应用的布局方向是否为从右到左（RTL）。
 *
 * 从 Android 4.2（API 级别 17，Jelly Bean MR1）开始，可以使用系统的 API 来检测布局方向。
 * 如果当前 API 级别小于 17，则默认返回 `false`。
 *
 * @return 如果布局方向为 RTL，则返回 `true`；否则返回 `false`。
 */
inline val Context.isLayoutRtl: Boolean
    get() = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) false else
        resources.configuration.layoutDirection == View.LAYOUT_DIRECTION_RTL


/**
 * 使用兼容的方式启动一个新的 Activity。
 * 此函数封装了启动 Activity 的逻辑，并允许设置启动选项和传递额外数据。
 *
 * @param activityClass 要启动的 Activity 的 Class 对象。
 * @param options 启动 Activity 时的动画效果 Bundle。
 * @param putExtra 一个 lambda 表达式，用于给启动的 Intent 添加额外的数据。
 * @throws IllegalStateException 如果系统中没有找到可以处理此 Intent 的 Activity。
 */
fun startActivityCompat(
    activityClass: Class<out Activity>,
    options: Bundle? = null,
    putExtra: (Intent.() -> Unit)? = null
) {
    val context = application
    val intent = Intent(context, activityClass)
    if (intent.resolveActivity(context.packageManager) != null) {
        putExtra?.invoke(intent)
        intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
        ActivityCompat.startActivity(context, intent, options)
    } else {
        throw IllegalStateException("No activity found to handle this intent.")
    }
}