package com.liecoder.framework.manager

import android.app.Activity
import android.app.Application
import android.os.Bundle
import com.liecoder.framework.model.AppStatus
import com.liecoder.framework.service.IAppStatusService

/**
 * LifecycleManager对象负责注册应用程序的生命周期回调。
 * 它使用内部的ActivityLifecycleCallbacks来监听和响应Activity的生命周期事件。
 */
object LifecycleManager {

    // 用于追踪活动启动的数量，以确定应用程序是处于前台还是后台。
    private var activityStartCount = 0

    /**
     * 注册应用程序状态服务，以便接收生命周期事件的通知。
     * @param mService 实现IAppStatusService接口的应用程序服务。
     */
    fun register(application: Application, mService: IAppStatusService) {
        application.registerActivityLifecycleCallbacks(object : Application
        .ActivityLifecycleCallbacks {
            override fun onActivityPaused(activity: Activity) {
            }

            override fun onActivityResumed(activity: Activity) {
            }

            override fun onActivityDestroyed(activity: Activity) {
                ActivityManager.pop(activity)
            }

            override fun onActivitySaveInstanceState(
                activity: Activity,
                outState: Bundle
            ) {
            }

            override fun onActivityCreated(
                activity: Activity,
                savedInstanceState: Bundle?
            ) {
                ActivityManager.push(activity)
            }

            override fun onActivityStarted(activity: Activity) {
                // 活动启动时的回调，增加启动计数
                activityStartCount++
                if (activityStartCount == 1) {
                    // 如果是首个活动启动，则通知应用程序进入前台状态
                    mService.onAppStatusChanged(AppStatus.Foreground, activity)
                }
            }

            override fun onActivityStopped(activity: Activity) {
                // 活动停止时的回调，减少启动计数
                activityStartCount--
                if (activityStartCount == 0) {
                    // 如果所有活动都已停止，则通知应用程序进入后台状态
                    mService.onAppStatusChanged(AppStatus.Background, activity)
                }
            }

        })
    }
}