package com.monmonkeygroup.openapi.client;

import com.monmonkeygroup.openapi.OpenApiException;
import com.monmonkeygroup.openapi.protocol.IProtocol;
import com.monmonkeygroup.openapi.protocol.Packet;
import org.java_websocket.client.WebSocketClient;
import org.java_websocket.framing.CloseFrame;
import org.java_websocket.handshake.ServerHandshake;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.URI;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;

public class WebsocketConn extends WebSocketClient implements IClientConn {
    private static final Logger log = LoggerFactory.getLogger(WebsocketConn.class);
    private Set<ISocketServiceListener> listeners = ConcurrentHashMap.newKeySet();
    private final IProtocol<String> protocol;

    public WebsocketConn(URI uri, IProtocol<String> protocol) {
        super(uri);
        this.protocol = protocol;
    }

    public void init() throws OpenApiException {
        try {
            if (!this.connectBlocking(15, TimeUnit.SECONDS)) {
                throw new OpenApiException(0, "connect timeout");
            }
        } catch (InterruptedException e) {
            throw new OpenApiException(0, e.getMessage());
        }
    }

    public boolean isOpen() {
        return super.isOpen();
    }

    @Override
    public void sendPacket(Packet packet) {
        String s = protocol.pack(packet);
        log.debug("WriteMessage: {}", s);
        synchronized (this) {
            send(s);
        }
    }

    @Override
    public boolean reconnectSocket() throws Exception {
        reconnect();
        return true;
    }

    public void close(String reason) {
        this.close(CloseFrame.NORMAL, reason);
    }

    @Override
    public void onOpen(ServerHandshake serverHandshake) {
        for (ISocketServiceListener listener : listeners) {
            listener.onConnected();
        }
    }

    @Override
    public void onMessage(String s) {
        log.debug("ReadMessage: {}", s);
        Packet packet;
        try {
            packet = protocol.unpack(s);
            if (null == packet) {
                return;
            }
        } catch (Exception e) {
            throw new RuntimeException(e);
        }

        for (ISocketServiceListener listener : listeners) {
            listener.onMessage(packet);
        }
    }

    @Override
    public void onClose(int i, String s, boolean b) {
        log.debug("close conn, err: {},{}", i, s);
        for (ISocketServiceListener listener : listeners) {
            if (b) {
                listener.onClose("close by client, " + s);
            } else {
                listener.onClose("close conn, " + s);
            }
        }
    }

    @Override
    public void onError(Exception e) {
        log.error(e.getMessage(), e);
    }

    @Override
    public void addListener(ISocketServiceListener listener) {
        listeners.add(listener);
    }

}
