package pluginloader.api

import kotlinx.serialization.SerialName
import kotlinx.serialization.Serializable
import org.bukkit.Material
import org.bukkit.enchantments.Enchantment
import org.bukkit.inventory.ItemFlag
import org.bukkit.inventory.ItemStack
import pluginloader.api.bukkit.NBT
import java.util.*

@Serializable
class Item: Cloneable{
    constructor()

    /**
     * Work like Item().apply{}
     */
    constructor(callback: Item.() -> Unit){callback(this)}

    var type: Material = Material.STONE
    var amount: Int = 1
    var damage: Int = 0
    var name: String = ""
    val lore: ArrayList<String> = ArrayList(0)
    val enchantment: MutableMap<@Serializable(EnchantmentSerializer::class) Enchantment, Int> = HashMap(0)
    @SerialName("data")
    var nbt: NBT = NBT.new()
    var unbreakable: Boolean = false
    var flags: MutableSet<ItemFlag> = EnumSet.noneOf(ItemFlag::class.java)

    /**
     * Build [ItemStack] with name, lore, etc
     */
    fun item(): ItemStack{
        val item = ItemStack(type, amount, damage.toShort())
        if(type == Material.AIR)return item
        val meta = item.itemMeta!!
        if(name.isNotEmpty())meta.setDisplayName(name)
        if(lore.isNotEmpty())meta.lore = lore
        enchantment.forEach{meta.addEnchant(it.key, it.value, true)}
        meta.isUnbreakable = unbreakable
        flags.forEach(meta::addItemFlags)
        item.itemMeta = meta
        this.nbt.writeTo(item)
        return item
    }

    companion object{
        /**
         * Build [Item] with data from [stack]
         * NBT data can be filled with enchantments, display and other
         */
        fun item(stack: ItemStack): Item{
            val item = Item()
            item.type = stack.type
            item.amount = stack.amount
            item.damage = stack.durability.toInt()
            if(stack.hasItemMeta()){
                val meta = stack.itemMeta!!
                if(meta.hasDisplayName())item.name = meta.displayName
                if(meta.hasLore())item.lore.addAll(meta.lore!!)
                if(meta.hasEnchants())item.enchantment.putAll(meta.enchants)
                item.flags.addAll(meta.itemFlags)
                item.unbreakable = meta.isUnbreakable
                item.nbt = NBT.clone(stack)
            }
            return item
        }

        @Deprecated("This is useless", ReplaceWith("Item()", "pluginloader.api.Item"))
        fun default() = Item()
    }

    fun amount(new: Int): Item = apply{amount = new}
    fun damage(new: Int): Item = apply{damage = new}
    fun type(new: Material): Item = apply{type = new}
    fun name(new: String): Item = apply{name = new}
    fun unbreakable(): Item = apply{unbreakable = true}
    fun lore(vararg lores: String): Item = apply{lore.addAll(lores)}
    fun replaceLore(vararg lores: String): Item = apply{lore.clear();lore.addAll(lores)}
    fun enchantment(ench: Enchantment, level: Int): Item = apply{enchantment[ench] = level}
    fun flag(flag: ItemFlag): Item = apply{flags.add(flag)}
    fun flags(vararg flag: ItemFlag): Item = apply{flags.addAll(flag)}

    /**
     * Replace [from] to [to] in name and lore
     */
    fun replaceText(from: String, to: String): Item{
        return replaceText{replace(from, to)}
    }

    /**
     * Run [callback] on name and every lore line
     */
    fun replaceText(callback: String.() -> String): Item{
        name = callback(name)
        lore.replaceAll(callback)
        return this
    }

    fun hasNbt(key: String): Boolean = this.nbt.has(key)
    fun stringNbt(key: String): String = this.nbt.string(key, "") ?: ""
    fun intNbt(key: String): Int = this.nbt.int(key, 0)
    fun doubleNbt(key: String): Double = this.nbt.double(key, 0.0)
    operator fun set(key: String, value: String) = this.nbt.setString(key, value)
    operator fun set(key: String, value: Int) = this.nbt.setInt(key, value)
    operator fun set(key: String, value: Double) = this.nbt.setDouble(key, value)

    override fun clone(): Item = Item().let{copyTo(it);it}
    fun copy(): Item = Item().let{copyTo(it);it}

    fun copyTo(item: Item){
        item.amount = amount
        item.damage = damage
        item.type = type
        item.name = name
        item.lore.addAll(lore)
        item.enchantment.putAll(enchantment)
        item.nbt = nbt.clone()
        item.unbreakable = unbreakable
        item.flags.addAll(flags)
    }

    override fun toString(): String {
        return "Item(type=$type, amount=$amount, damage=$damage, name='$name', lore=$lore, enchantment=$enchantment, nbt=$nbt, unbreakable=$unbreakable, flags=$flags)"
    }

    override fun equals(other: Any?): Boolean {
        if (this === other) return true
        if (javaClass != other?.javaClass) return false

        other as Item

        if (type != other.type) return false
        if (amount != other.amount) return false
        if (damage != other.damage) return false
        if (name != other.name) return false
        if (lore != other.lore) return false
        if (enchantment != other.enchantment) return false
        if (this.nbt != other.nbt) return false
        if (unbreakable != other.unbreakable) return false
        if (flags != other.flags) return false

        return true
    }

    override fun hashCode(): Int {
        var result = type.hashCode()
        result = 31 * result + amount
        result = 31 * result + damage
        result = 31 * result + name.hashCode()
        result = 31 * result + lore.hashCode()
        result = 31 * result + enchantment.hashCode()
        result = 31 * result + nbt.hashCode()
        result = 31 * result + unbreakable.hashCode()
        result = 31 * result + flags.hashCode()
        return result
    }
}