package pluginloader.api

import org.bukkit.Bukkit
import org.bukkit.command.CommandSender
import org.bukkit.event.Event
import org.bukkit.event.EventPriority
import org.bukkit.scheduler.BukkitTask
import kotlin.reflect.KClass

interface Plugin: LoaderPlugin {
    /**
     * Register bukkit command with [name]
     * See also [Command]
     * ```
     * plugin.registerCommand("hi", {sender, args -> sender.sendMessage("hi")})
     * ```
     * @param name command name, /hi
     * @param checkOp if true, only player.isOp can call this command
     * @param sender [Sender] or [CommandSender]
     * @param args [Args] /example a b -> arrayOf("a", "b")
     */
    fun registerCommand(name: String, callback: (Sender, Args) -> Unit, checkOp: Boolean = false, vararg aliases: String)

    /**
     * Register tab complete on [registerCommand]
     * If command not registered, doing nothing
     * ```
     * plugin.registerTabComplete
     * ```
     * @param name command name, /hi
     * @param callback call on tab complete
     */
    fun registerTabComplete(name: String, callback: (Sender, Args) -> List<String>, autoSort: Boolean = true)

    /**
     * Register bukkit listener on class [kClass]
     * ```
     * plugin.registerListener(PlayerJoinEvent::class, {event -> event.player.sendMessage("hi")})
     * ```
     *
     * @param T event class
     * @param priority call sequence: LOW (first), NORMAL (second), HIGH (third)
     * @param ignoreCancelled if true and event cancelled, [listener] don't call
     */
    fun <T: Event> registerListener(kClass: KClass<T>, listener: (event: T) -> Unit, priority: EventPriority = EventPriority.NORMAL, ignoreCancelled: Boolean = false)

    /**
     * Calling [handler] once on unload plugin with /plu u example or on stop server
     */
    override fun unloadHandler(handler: () -> Unit)

    fun runTaskTimer(time: Int, callback: () -> Unit){
        runTaskTimer(time, callback, false)
    }

}

/**
 * Register bukkit listener on class [T]
 * ```
 * plugin.listener<PlayerJoinEvent>{player.sendMessage("hi")}
 * ```
 *
 * @param T event class
 * @param priority call sequence: LOW (first), NORMAL (second), HIGH (third)
 * @param ignoreCancelled if true and event cancelled, [listener] don't call
 */
inline fun <reified T: Event> Plugin.listener(priority: EventPriority = EventPriority.NORMAL, ignoreCancelled: Boolean = false, crossinline listener: T.() -> Unit){
    registerListener(T::class, {listener(it)}, priority, ignoreCancelled)
}

/**
 * Register bukkit command with [name]
 * See also [Command]
 * ```
 * plugin.registerCommand("hi", {sender, args -> sender.sendMessage("hi")})
 * ```
 * @param name command name, /hi
 * @param checkOp if true, only player.isOp can call this command
 * @param sender [Sender] or [CommandSender]
 * @param args [Args] /example a b -> arrayOf("a", "b")
 */
@Suppress("NOTHING_TO_INLINE")
inline fun Plugin.command(checkOp: Boolean = false, aliases: Array<String> = emptyArray(), name: String, noinline callback: (sender: Sender, args: Args) -> Unit){
    registerCommand(name, callback, checkOp, *aliases)
}

/**
 * Call [callback] in main thread after [time] ticks (1/20 second)
 */
fun LoaderPlugin.runTaskLater(time: Int, callback: () -> Unit, callOnUnload: Boolean = false){
    runAbstractTask({Bukkit.getScheduler().runTaskLater(plugin, it, time.toLong())}, callback, callOnUnload, true)
}

/**
 * Call [callback] async after [time] ticks (1/20 second)
 */
fun LoaderPlugin.runAsyncLater(time: Int, callback: () -> Unit, callOnUnload: Boolean = false){
    runAbstractTask({Bukkit.getScheduler().runTaskLaterAsynchronously(plugin, it, time.toLong())}, callback, callOnUnload, true)
}

/**
 * Call [callback] in main thread with period [time] ticks (1/20 second)
 * Task cancel after plugin unload
 */
fun LoaderPlugin.runTaskTimer(time: Int, callback: () -> Unit, callOnUnload: Boolean = false){
    runAbstractTask({Bukkit.getScheduler().runTaskTimer(plugin, it, time.toLong(), time.toLong())}, callback, callOnUnload, false)
}

/**
 * Call [callback] async with period [time] ticks (1/20 second)
 * Task cancel after plugin unload
 */
fun LoaderPlugin.runAsyncTimer(time: Int, callback: () -> Unit, callOnUnload: Boolean = false){
    runAbstractTask({Bukkit.getScheduler().runTaskTimerAsynchronously(plugin, it, time.toLong(), time.toLong())}, callback, callOnUnload, false)
}

private fun LoaderPlugin.runAbstractTask(task: (() -> Unit) -> BukkitTask, callback: () -> Unit, callOnUnload: Boolean, singleCall: Boolean){
    var unload: (() -> Unit)? = null
    val bukkitTask = if(singleCall) task {
        unload.nonNull(this::removeUnloadHandler)
        callback()
    } else task(callback)
    unload = {
        if(!bukkitTask.isCancelled) {
            bukkitTask.cancel()
            if (callOnUnload) callback()
        }
    }
    unloadHandler(unload)
}