package pluginloader.api.bukkit

import kotlinx.serialization.KSerializer
import kotlinx.serialization.Serializable
import kotlinx.serialization.builtins.serializer
import kotlinx.serialization.descriptors.SerialDescriptor
import kotlinx.serialization.encoding.Decoder
import kotlinx.serialization.encoding.Encoder
import kotlinx.serialization.json.*
import org.bukkit.inventory.ItemStack
import pluginloader.internal.bukkit.AbstractItemNBT

@JvmInline
@Serializable(NBT.Serializer::class)
value class NBT private constructor(private val nbt: Any){
    companion object{
        /**
         * @return Read only nbt data from [this]
         */
        fun ItemStack?.readNBT(): NBT = read(this)
        /**
         * @return Mutable nbt data from [this]
         */
        fun ItemStack.writeNBT(): NBT = write(this)

        /**
         * @return Read only nbt data from [item]
         */
        fun read(item: ItemStack?): NBT = NBT(provider().read(item))
        /**
         * @return Mutable nbt data from [item]
         */
        fun write(item: ItemStack): NBT = NBT(provider().write(item))
        /**
         * @return Cloned nbt data from [item]
         */
        fun clone(item: ItemStack): NBT = NBT(provider().cloneFromItem(item))
        /**
         * @return Create empty nbt data
         */
        fun new(): NBT = NBT(provider().new())

        fun has(item: ItemStack, key: String) = read(item).has(key)
        @Suppress("NOTHING_TO_INLINE")
        inline fun hasNot(item: ItemStack, key: String) = has(item, key).not()
        fun remove(item: ItemStack, key: String) = write(item).remove(key)

        fun string(item: ItemStack, key: String, default: String? = null) = read(item).string(key, default)
        fun int(item: ItemStack, key: String, default: Int = -1) = read(item).int(key, default)
        fun double(item: ItemStack, key: String, default: Double = -1.0) = read(item).double(key, default)
        fun long(item: ItemStack, key: String, default: Long = -1) = read(item).long(key, default)

        fun setString(item: ItemStack, key: String, value: String) = write(item).setString(key, value)
        fun setInt(item: ItemStack, key: String, value: Int) = write(item).setInt(key, value)
        fun setDouble(item: ItemStack, key: String, value: Double) = write(item).setDouble(key, value)
        fun setLong(item: ItemStack, key: String, value: Long) = write(item).setLong(key, value)

        //Don't delete, binary comparable
        internal fun fromJson(json: JsonObject): Any = provider().fromJson(json)
        fun fromJsonNBT(json: JsonObject): NBT = NBT(fromJson(json))

        @Suppress("NOTHING_TO_INLINE")
        private inline fun provider(): AbstractItemNBT{
            return AbstractItemNBT.provider
        }
    }

    fun has(key: String): Boolean = provider().has(nbt, key)
    @Suppress("NOTHING_TO_INLINE")
    inline fun hasNot(key: String): Boolean = has(key).not()
    fun remove(key: String): Boolean = provider().remove(nbt, key)
    fun clone(): NBT = NBT(provider().clone(nbt))
    fun writeTo(item: ItemStack): Unit = provider().writeTo(nbt, item)

    fun string(key: String, default: String?): String? = provider().string(nbt, key, default)
    fun int(key: String, default: Int): Int = provider().int(nbt, key, default)
    fun double(key: String, default: Double): Double = provider().double(nbt, key, default)
    fun long(key: String, default: Long): Long = provider().long(nbt, key, default)

    fun setString(key: String, value: String): Unit = provider().setString(nbt, key, value)
    fun setInt(key: String, value: Int): Unit = provider().setInt(nbt, key, value)
    fun setDouble(key: String, value: Double): Unit = provider().setDouble(nbt, key, value)
    fun setLong(key: String, value: Long): Unit = provider().setLong(nbt, key, value)

    fun toJson(): JsonObject = provider().toJson(nbt)

    override fun toString(): String {
        return toJson().toString()
    }

    object Serializer: KSerializer<NBT>{
        override val descriptor: SerialDescriptor = String.serializer().descriptor

        override fun deserialize(decoder: Decoder): NBT {
            if(decoder is JsonDecoder){
                val element = decoder.decodeJsonElement()
                return if(element is JsonObject) fromJsonNBT(element)

                else fromJsonNBT(Json.decodeFromString(JsonObject.serializer(), (element as JsonPrimitive).content))
            }
            return fromJsonNBT(Json.decodeFromString(JsonObject.serializer(), decoder.decodeString()))
        }

        override fun serialize(encoder: Encoder, value: NBT) {
            if(encoder is JsonEncoder) {
                JsonObject.serializer().serialize(encoder, (value).toJson())
                return
            }
            encoder.encodeString(Json.encodeToString(JsonObject.serializer(), value.toJson()))
        }
    }
}