package pluginloader.gradle

import com.jcraft.jsch.ChannelSftp
import com.jcraft.jsch.JSch
import com.jcraft.jsch.agentproxy.ConnectorFactory
import com.jcraft.jsch.agentproxy.RemoteIdentityRepository
import org.gradle.api.Plugin
import org.gradle.api.Project
import org.gradle.jvm.tasks.Jar
import java.io.File
import java.nio.file.Files

class GradlePlugin: Plugin<Project> {
    override fun apply(project: Project) {
        //Apply plugins
        project.applyPlugin("org.jetbrains.kotlin.jvm")
        project.applyPlugin("org.jetbrains.kotlin.plugin.serialization")
        //Used on publish
        project.applyPlugin("maven-publish")
        //Used on publish to maven central
        project.applyPlugin("signing")

        //Maven central used anyway, because api jars there
        project.repositories.add(project.repositories.mavenCentral())

        //Some libs bundled with pluginloader jar
        project.compileDependency("org.jetbrains.kotlin:kotlin-stdlib:$kotlinVersion")
        project.compileDependency("org.jetbrains.kotlinx:kotlinx-serialization-core:$kotlinSerializationVersion")
        project.compileDependency("org.jetbrains.kotlinx:kotlinx-serialization-json:$kotlinSerializationVersion")
        project.compileDependency("io.github.pluginloader:api:$pluVersion")

        //Special configuration for
        val depConf = project.createConfiguration("dependency")
        val depMavenConf = project.createConfiguration("mavenDependency")

        //Add api to gradle
        val config = Config(project)
        project.extensions.add("plu", config)

        //Write information about dependencies
        project.tasks.getByPath("jar").apply jar@ {
            this as Jar
            archiveFileName.set("${project.name}.jar")

            doFirst{
                //Delete older builds data
                project.file(hackyPath).apply{if(exists()) deleteRecursively()}

                //Create files for dependency plugins
                project.addToJarPerDependency(depConf, writeToFile = {
                    "$group:$name:$version"
                }){"$name.dependency"}

                //For mavenDependency plugins
                project.addToJarPerDependency(depMavenConf){"$name.dependency"}

                //Mark expand
                config.expand?.apply{project.addFileToJar("$this.expand")}
            }
        }

        //Auto update project name & package, very useful for templates
        project.file("plu_project.txt").apply update@ {
            val path = toPath()
            //'Files' functions work more stable on Windows
            if(Files.notExists(path))return@update

            //Old project name in plu_project.txt, expect something like 'template'
            val oldName = readText().split("\n")[0]
            //New name, expect new project name
            val newName = project.file(".").name

            //I hope, if name contains 'template', is mean this is template
            if(newName.contains("template"))return@update

            //Remove template mark
            Files.delete(path)

            //Replace package in kotlin
            val src = project.file("src/main/kotlin/Plugin.kt")
            val srcPath = src.toPath()
            if(Files.exists(srcPath)) src.writeText(src.readText().replace(oldName, newName.replace("-", "_")))

            //Replace project name in gradle
            val settings = project.file("./settings.gradle.kts")
            val settingsPath = settings.toPath()
            if(Files.exists(settingsPath)) settings.writeText(settings.readText().replace(oldName, newName))
        }

        //Custom task for easy uploading plugins to test production
        //Require PLU_PUSH_HOST & PLU_PUSH_USER & PLU_PUSH_PORT in env or gradle.settings
        //And PageAnt or any SSH agent
        project.task("upload"){
            //Build plugin before upload
            it.dependsOn("build")

            it.doLast{
                //Get some information
                val host = getenv(project, "${config.prefix}_PUSH_HOST") ?:
                    error("Can't find env ${config.prefix}_PUSH_HOST, example: 'example.com', also can set ${config.prefix}_PUSH_PORT, example: '22'")
                val user = getenv(project, "${config.prefix}_PUSH_USER") ?: error("Can't find env ${config.prefix}_PUSH_USER, example: 'user'")
                val port = getenv(project, "${config.prefix}_PUSH_PORT")?.toInt() ?: 22
                val uploadFrom = "build/libs/${project.name}.jar"
                val uploadTo = project.properties["p"] ?: getenv(project, "${config.prefix}_PUSH_DIR") ?:
                    error("Can't find env ${config.prefix}_PUSH_DIR, example: 'dir/dir', also can use 'gradle upload -Pp=path/to/dir'")

                //JSch is magic, feel free to copy & paste code below
                val jsch = JSch()
                //Set known hosts from ~/.ssh/known_hosts
                jsch.setKnownHosts("${System.getProperty("user.home")}${File.separator}.ssh${File.separator}known_hosts")
                //Set remote identity
                jsch.setIdentityRepository(RemoteIdentityRepository(ConnectorFactory.getDefault().createConnector()))

                val session = jsch.getSession(user, host, port)
                session.connect()

                val sftp = session.openChannel("sftp") as ChannelSftp
                sftp.connect()
                //Absolute path just in case
                sftp.put(project.file(uploadFrom).absolutePath, uploadTo.toString())
                sftp.disconnect()
                session.disconnect()
            }
        }

    }
}