package pluginloader.gradle

import org.gradle.api.Project
import org.gradle.api.artifacts.dsl.RepositoryHandler
import org.gradle.api.file.DuplicatesStrategy
import org.gradle.api.plugins.JavaPluginExtension
import org.gradle.api.publish.maven.MavenPublication
import org.gradle.plugins.signing.SigningExtension

open class Config(private val project: Project){
    internal var prefix = "PLU"
    internal var expand: String? = null

    /**
     * Include official plugins, like configs, spi, something else
     */
    fun central(vararg plugins: String){
        project.addDependencies("dependency", plugins){"io.github.pluginloader:${if(it.contains(":")) it else "$it:+"}"}
    }

    /**
     * Add dependency(pluginloader:$it)
     * Can be used in private projects
     */
    fun plu(vararg plugins: String){
        project.addDependencies("dependency", plugins){"pluginloader:${if(it.contains(":")) it else "$it:+"}"}
    }

    /**
     * Add dependencies, include to jar if use includeDep()
     */
    fun impl(vararg dep: String): Unit = project.addDependencies("implementation", dep)

    /**
     * Add dependencies in compile time
     */
    fun compileOnly(vararg dep: String): Unit = project.addDependencies("compileOnly", dep)

    /**
     * Add minecraft papermc dependencies
     *
     * @param version minecraft version, probably 1.12.2 or newest
     */
    fun paper(version: String){
        project.repositories.add(project.repositories.maven{it.url = project.uri("https://papermc.io/repo/repository/maven-public/")})
        project.compileDependency("io.github.pluginloader:bukkit-api:$pluVersion")
        if(version == "1.12.2"){
            project.compileDependency("com.destroystokyo.paper:paper-api:$version-R0.1-SNAPSHOT")
        }else{//1.13+?
            project.compileDependency("io.papermc.paper:paper-api:$version-R0.1-SNAPSHOT")
        }
    }

    /**
     * Allow adding private repository to project
     * Require env $prefix_NEXUS_URL, $prefix_NEXUS_USER, $prefix_NEXUS_PASSWORD
     *
     * @param prefix - used as prefix in env
     */
    fun private(prefix: String = "PLU") {
        fun findOrError(name: String): String{
            return getenv(project, "${prefix}_NEXUS_$name") ?: error("Can't add private repo, env ${prefix}_NEXUS_$name not found")
        }

        val envUrl = project.uri(findOrError("URL"))
        val user = findOrError("USER")
        val password = findOrError("PASSWORD")

        //Used in task upload, see GradlePlugin
        this.prefix = prefix

        fun RepositoryHandler.applyMaven(){
            maven {
                it.url = envUrl

                it.credentials { credentials ->
                    credentials.username = user
                    credentials.password = password
                }
            }
        }

        //Add to repositories{}
        project.repositories.applyMaven()

        //I don't find any way better to get plugin
        val publishing = (project as org.gradle.api.plugins.ExtensionAware)
            .extensions.getByName("publishing") as org.gradle.api.publish.PublishingExtension

        //Generate maven publishing data
        publishing.publications.create("java", MavenPublication::class.java){
            it.from(project.components.getByName("java"))
        }

        //Add publish repository
        publishing.repositories.applyMaven()
    }

    /**
     * Include impl dependencies to jar
     */
    fun includeDep(){
        project.tasks.withType(org.gradle.api.tasks.bundling.Jar::class.java){
            it.duplicatesStrategy = DuplicatesStrategy.INCLUDE
            it.from(project.configurations.getByName("runtimeClasspath").map{map -> if(map.isDirectory) map else project.zipTree(map)})
        }
    }

    /**
     * Do not use, very specific thing, very unstable
     * Reorder loading order, guaranteed load plugin after @Load and before @PostExpand
     * TODO: Must be documented later
     */
    fun expand(plugin: String){
        expand = plugin
    }

    /**
     * Just ignore this method
     * Calling only in pluginloader/mono
     */
    fun public() {
        //Build plugins from pluginloader/mono everywhere
        if (getenv(project, "MAVEN_CENTRAL_USER") == null
            || getenv(project, "MAVEN_CENTRAL_PASSWORD") == null) return

        val publishing = (project as org.gradle.api.plugins.ExtensionAware)
            .extensions.getByName("publishing") as org.gradle.api.publish.PublishingExtension

        val java = (project as org.gradle.api.plugins.ExtensionAware)
            .extensions.getByName("java") as JavaPluginExtension

        java.withSourcesJar()
        java.withJavadocJar()

        //Useless shit, but required by maven central
        publishing.publications.create("maven", MavenPublication::class.java){
            it.from(project.components.getByName("java"))
            it.pom {m ->
                m.name.set("Pluginloader")
                m.description.set("One of many Pluginloader plugins")
                m.url.set("https://github.com/pluginloader/mono")

                m.licenses {l ->
                    l.license {lic ->
                        lic.name.set("MIT")
                        lic.url.set("https://mit-license.org/")
                    }
                }
                m.developers {d ->
                    d.developer {dev ->
                        dev.id.set("6oogle")
                        dev.name.set("6oogle")
                        dev.email.set("mail@example.com")
                    }
                }
                m.scm {scm ->
                    scm.connection.set("scm:git:https://github.com/pluginloader/mono.git")
                    scm.developerConnection.set("scm:git:https://github.com/pluginloader/mono.git")
                    scm.url.set("https://github.com/pluginloader/mono")
                }
            }
        }

        //Provide maven central repo
        publishing.repositories {
            it.maven {repo ->
                repo.url = project.uri("https://s01.oss.sonatype.org/service/local/staging/deploy/maven2/")
                repo.credentials {cre ->
                    cre.username = getenv(project, "MAVEN_CENTRAL_USER")!!
                    cre.password = getenv(project, "MAVEN_CENTRAL_PASSWORD")!!
                }
            }
        }

        //Sometimes signing work strange
        val signing = (project as org.gradle.api.plugins.ExtensionAware)
            .extensions.getByName("signing") as SigningExtension
        signing.useGpgCmd()
        signing.sign(publishing.publications.getByName("maven"))
    }
}